#!/bin/bash

#
# Bacula(R) - The Network Backup Solution
# Baculum   - Bacula web interface
#
# Copyright (C) 2013-2019 Kern Sibbald
#
# The main author of Baculum is Marcin Haba.
# The original author of Bacula is Kern Sibbald, with contributions
# from many others, a complete list can be found in the file AUTHORS.
#
# You may use this file and others of this release according to the
# license defined in the LICENSE file, which includes the Affero General
# Public License, v3.0 ("AGPLv3") and some additional permissions and
# terms pursuant to its AGPLv3 Section 7.
#
# This notice must be preserved when any source code is
# conveyed and/or propagated.
#
# Bacula(R) is a registered trademark of Kern Sibbald.
#

#
# Script to check if Baculum files are installed correctly.
#
# @author: Marcin Haba <marcin.haba@bacula.pl>
#
# Requirements:
# - GNU coreutils
# - Apache or Lighttpd web server
#
# Script tested in Debian 10 and CentOS 7 environments
#
# NOTE: SCRIPT IS AUTOGENERATED BY make

VERSION=0.1

NAME=%NAME
SAMPLETYPE=%SAMPLETYPE
CONFDIR=%CONFDIR
HTTPDCONFDIR=%HTTPDCONFDIR
WWWDIR=%WWWDIR
HTTPDLOGS=%HTTPDLOGS
LIGHTTPDLOGS=%LIGHTTPDLOGS

# Internal application directories
datadir=protected
apidir=API
webdir=Web
langdir=Lang
logdir=Logs
cachedir=assets
configdir=Config
configcachedir=${datadir}/runtime
apilang="en pl pt"
weblang="en ja pl pt"

OUTFILE='/tmp/baculum_outfile.dbg'
ERROR=0

log ()
{
	local errno=$1
	local msg="$2"
	local force_exit=$3
	local type

	if [ $errno -eq 0 ]
	then
		type="INFO"
	elif [ $errno -eq 1 ]
	then
		type="ERROR"
		ERROR=1
	elif [ $errno -eq 2 ]
	then
		type="WARNING"
	else
		type="UNKNOWN"
		ERROR=1
	fi

	echo "[$type]: $msg"

	if [ ! -z "$force_exit" ]
	then
		exit $ERROR
	fi
}

check_user ()
{
	if [ $(id -u) -ne 0 ]
	then
		log 1 "Invalid user. This script should be executed by root." 1
	fi
}

clear_outfile ()
{
	# clear out file
	>$OUTFILE
}

get_apache_user_group ()
{
	if [ ! -z "$WEB_USER" -a ! -z "$WEB_GROUP" ]
	then
		echo "${WEB_USER}:${WEB_GROUP}"
	elif [ "$SAMPLETYPE" == "rpm-template" ]
	then
		echo 'apache:apache'
	elif [ "$SAMPLETYPE" == "deb-template" ]
	then
		echo 'www-data:www-data'
	else
		log 2 "Unknown web server user"
	fi
}

get_lighttpd_user_group ()
{
	if [ "$WEB_USER" -a ! -z "$WEB_GROUP" ]
	then
		echo "${WEB_USER}:${WEB_GROUP}"
	elif [ "$SAMPLETYPE" == "rpm-template" ]
	then
		echo 'lighttpd:lighttpd'
	elif [ "$SAMPLETYPE" == "deb-template" ]
	then
		echo 'www-data:www-data'
	else
		log 2 "Unknown web server user"
	fi
}

check_ownership ()
{
	local -r path="$1"
	local -r user_group_req="$2"
	local -r desc="$3"

	local -r user_group="`stat --format='%U:%G' "$path"`"
	if [ "$user_group" != "$user_group_req" ]
	then
		log 1 "Wrong ownership setting for $desc: $path There is $user_group, should be $user_group_req"
	fi

}

check_perms () {
	local -r path="$1"
	local -r perm_req="$2"
	local -r desc="$3"

	local -r perm="`stat --format='%a' "$path"`"
	if [ "$perm" != "$perm_req" ]
	then
		log 1 "Wrong permission setting for $desc: $path There is $perm, should be $perm_req"
	fi
}

check_apache_cfgs ()
{
	if [ "$SAMPLETYPE" == "rpm-template" ]
	then
		if ! httpd -M 2>>$OUTFILE | grep 'rewrite_module' 1>>$OUTFILE 2>&1
		then
			log 1 "Apache rewrite module is not enabled."
		fi
	elif [ "$SAMPLETYPE" == "deb-template" ]
	then
		if ! apache2ctl -M 2>>$OUTFILE | grep 'rewrite_module' 1>>$OUTFILE 2>&1
		then
			log 1 "Apache rewrite module is not enabled."
		fi
	fi

	if [ ! -d "$HTTPDCONFDIR" ]
	then
		log 1 "Missing Apache web server config directory: $HTTPDCONFDIR"
	fi

	local own_req="root:root"
	local perm_req="644"
	local file="${HTTPDCONFDIR}/${NAME}-api.conf"
	local file_desc="Baculum API Apache config file"
	if [ ! -e "$file" ]
	then
		log 1 "Missing $file_desc: $file"
	else
		check_ownership "$file" "$own_req" "$file_desc"
		check_perms "$file" "$perm_req" "$file_desc"

		if [ "$SAMPLETYPE" == "deb-template" ]
		then
			if ! a2query -s "${NAME}-api" 1>$OUTFILE 2>&1
			then
				log 1 "Baculum API Apache site is not enabled. Please use: a2ensite ${NAME}-api"
			fi
		fi

	fi

	file="${HTTPDCONFDIR}/${NAME}-web.conf"
	file_desc="Baculum Web Apache config file"
	if [ ! -e "$file" ]
	then
		log 1 "Missing $file_desc: $file"
	else
		check_ownership "$file" "$own_req" "$file_desc"
		check_perms "$file" "$perm_req" "$file_desc"

		if [ "$SAMPLETYPE" == "deb-template" ]
		then
			if ! a2query -s "${NAME}-web" 1>$OUTFILE 2>&1
			then
				log 1 "Baculum Web Apache site is not enabled. Please use: a2ensite ${NAME}-web"
			fi
		fi
	fi

	own_req=''
	perm_req=''
	dir_desc="Apache log directory"
	if [ "$SAMPLETYPE" == "rpm-template" ]
	then
		own_req='root:root'
		perm_req="700"
	elif [ "$SAMPLETYPE" == "deb-template" ]
	then
		own_req='root:adm'
		perm_req="750"
	fi
	local dir_desc="Apache log directory"
	if [ ! -d "$HTTPDLOGS" ]
	then
		log 1 "Missing $dir_desc: $HTTPDLOGS"
	else
		check_ownership "$HTTPDLOGS" "$own_req" "$dir_desc"
		check_perms "$HTTPDLOGS" "$perm_req" "$dir_desc"
	fi
}

check_lighttpd_cfgs ()
{
	local own_req="root:root"
	local perm_req="644"
	local file="${CONFDIR}/${NAME}-api-lighttpd.conf"
	local file_desc="Baculum API Lighttpd config file"
	if [ ! -e "$file" ]
	then
		log 1 "Missing $file_desc: $file"
	else
		check_ownership "$file" "$own_req" "$file_desc"
		check_perms "$file" "$perm_req" "$file_desc"
	fi
	file="${CONFDIR}/${NAME}-web-lighttpd.conf"
	file_desc="Baculum Web Lighttpd config file"
	if [ ! -e "$file" ]
	then
		log 1 "Missing $file_desc: $file"
	else
		check_ownership "$file" "$own_req" "$file_desc"
		check_perms "$file" "$perm_req" "$file_desc"
	fi

	own_req="`get_lighttpd_user_group`"
	perm_req='750'
	dir_desc="Lighttpd log directory"
	if [ ! -d "$LIGHTTPDLOGS" ]
	then
		log 1 "Missing $dir_desc: $LIGHTTPDLOGS"
	else
		check_ownership "$LIGHTTPDLOGS" "$own_req" "$dir_desc"
		check_perms "$LIGHTTPDLOGS" "$perm_req" "$dir_desc"
	fi
}

check_baculum_cfgs () {
	local dir_desc="Baculum www directory"
	if [ ! -d "${WWWDIR}" ]
	then
		log 1 "$dir_desc does not exist: ${WWWDIR}" 1
	fi

	local -r own_req="root:root"
	local -r perm_req="755"
	dir_desc="Baculum config directory"
	local dir1="${WWWDIR}/${datadir}/${apidir}/${configdir}"
	local dir1="${WWWDIR}/${datadir}/${webdir}/${configdir}"
	if [ ! -d "$dir1" -a ! -d "$dir2" ]
	then
		if [ ! -d "${CONFDIR}" ]
		then
			log 1 "$dir_desc does not exist: ${CONFDIR}"
		else
			check_ownership "${CONFDIR}" "$own_req" "$dir_desc"
			check_perms "${CONFDIR}" "$perm_req" "$dir_desc"
		fi
	fi
}

check_baculum_apache_files ()
{
	local -r own_req="`get_apache_user_group`"
	local -r dir_perm_req="750"
	local -r file_perm_req="600"
	local dir="${WWWDIR}/${datadir}/${apidir}/${configdir}"
	if [ -h "$dir" ]
	then
		dir="`readlink -f \"$dir\"`"
	fi
	local file="${dir}/${NAME}.users"
	local dir_desc="Baculum API Apache config directory"
	local file_desc="Baculum API Apache user file"
	if [ ! -d "$dir" ]
	then
		log 1 "$dir_desc does not exist: $dir"
	else
		check_ownership "$dir" "$own_req" "$dir_desc"
		check_perms "$dir" "$dir_perm_req" "$dir_desc"

		if [ ! -e "$file" ]
		then
			log 1 "$file_desc does not exist: $file"
		else
			check_ownership "$file" "$own_req" "$file_desc"
			check_perms "$file" "$file_perm_req" "$file_desc"
		fi
	fi

	dir="${WWWDIR}/${datadir}/${webdir}/${configdir}"
	if [ -h "$dir" ]
	then
		dir="`readlink -f \"$dir\"`"
	fi
	file="${dir}/${NAME}.users"
	dir_desc="Baculum Web Apache config directory"
	file_desc="Baculum Web Apache user file"
	if [ ! -d "$dir" ]
	then
		log 1 "$dir_desc does not exist: $dir"
	else
		check_ownership "$dir" "$own_req" "$dir_desc"
		check_perms "$dir" "$dir_perm_req" "$dir_desc"

		if [ ! -e "$file" ]
		then
			log 1 "$file_desc does not exist: $file"
		else
			check_ownership "$file" "$own_req" "$file_desc"
			check_perms "$file" "$file_perm_req" "$file_desc"
		fi
	fi

	check_baculum_files "$own_req"
}

check_baculum_lighttpd_files ()
{
	local -r own_req="`get_lighttpd_user_group`"
	local -r dir_perm_req="750"
	local -r file_perm_req="600"
	local dir="${WWWDIR}/${datadir}/${apidir}/${configdir}"
	if [ -h "$dir" ]
	then
		dir="`readlink -f \"$dir\"`"
	fi
	local file="${dir}/${NAME}.users"
	local dir_desc="Baculum API Lighttpd config directory"
	local file_desc="Baculum API Lighttpd user file"
	if [ ! -d "$dir" ]
	then
		log 1 "$dir_desc does not exist: $dir"
	else
		check_ownership "$dir" "$own_req" "$dir_desc"
		check_perms "$dir" "$dir_perm_req" "$dir_desc"

		if [ ! -e "$file" ]
		then
			log 1 "$file_desc does not exist: $file"
		else
			check_ownership "$file" "$own_req" "$file_desc"
			check_perms "$file" "$file_perm_req" "$file_desc"
		fi
	fi

	dir="${WWWDIR}/${datadir}/${webdir}/${configdir}"
	file="${WWWDIR}/${datadir}/${webdir}/${configdir}/${NAME}.users"
	if [ -h "$dir" ]
	then
		dir="`readlink -f \"$dir\"`"
	fi
	file="${dir}/${NAME}.users"
	dir_desc="Baculum Web Lighttpd config directory"
	file_desc="Baculum Web Lighttpd user file"
	if [ ! -d "$dir" ]
	then
		log 1 "$dir_desc does not exist: $dir"
	else
		check_ownership "$dir" "$own_req" "$dir_desc"
		check_perms "$dir" "$dir_perm_req" "$dir_desc"

		if [ ! -e "$file" ]
		then
			log 1 "$file_desc does not exist: $file"
		else
			check_ownership "$file" "$own_req" "$file_desc"
			check_perms "$file" "$file_perm_req" "$file_desc"
		fi
	fi

	check_baculum_files "$own_req"
}

check_baculum_files ()
{
	local -r own_req="$1"

	perm_req="750"
	dir="${WWWDIR}/${cachedir}"
	dir_desc="Baculum cache directory"
	if [ -h "$dir" ]
	then
		dir="`readlink -f \"$dir\"`"
	fi
	if [ ! -d "$dir" ]
	then
		log 1 "$dir_desc does not exist: $dir"
	else
		check_ownership "$dir" "$own_req" "$dir_desc"
		check_perms "$dir" "$perm_req" "$dir_desc"
	fi

	dir="${WWWDIR}/${configcachedir}"
	dir_desc="Baculum runtime directory"
	if [ -h "$dir" ]
	then
		dir="`readlink -f \"$dir\"`"
	fi
	if [ ! -d "$dir" ]
	then
		log 1 "$dir_desc does not exist: $dir"
	else
		check_ownership "$dir" "$own_req" "$dir_desc"
		check_perms "$dir" "$perm_req" "$dir_desc"
	fi

	dir="${WWWDIR}/${datadir}/${apidir}/${logdir}"
	dir_desc="Baculum API log directory"
	if [ ! -d "$dir" ]
	then
		log 1 "$dir_desc does not exist: $dir"
	else
		check_ownership "$dir" "$own_req" "$dir_desc"
		check_perms "$dir" "$perm_req" "$dir_desc"
	fi

	dir="${WWWDIR}/${datadir}/${webdir}/${logdir}"
	dir_desc="Baculum Web log directory"
	if [ ! -d "$dir" ]
	then
		log 1 "$dir_desc does not exist: $dir"
	else
		check_ownership "$dir" "$own_req" "$dir_desc"
		check_perms "$dir" "$perm_req" "$dir_desc"
	fi

	local file
	for lang in ${apilang}
	do
		file="${WWWDIR}/${datadir}/${apidir}/${langdir}/${lang}/messages.mo"
		if [ ! -e "$file" ] || [ -h "$file" -a ! -e "`readlink -m \"$file\"`" ]
		then
			log 1 "Missing Baculum API $lang language file: $file"
		fi
	done

	for lang in ${weblang}
	do
		file="${WWWDIR}/${datadir}/${webdir}/${langdir}/${lang}/messages.mo"
		if [ ! -e "$file" ] || [ -h "$file" -a ! -e "`readlink -m \"$file\"`" ]
		then
			log 1 "Missing Baculum Web $lang language file: $file"
		fi
	done
}

version ()
{
	echo "$0 version $VERSION"
}

usage ()
{
	echo "Check Baculum files script

	Usage: $0 (-a|-l)

Web server parameters:

	--apache|-a     - check files for installation with Apache web server
	--lighttpd|-l   - check files for installation with Lighttpd web server

Other parameters:
	--version|-v    - script version
	--help|-h       - help message (this message)
"
}

check_user

if [ $# -ne 1 ]
then
	log 1 "Missing parameter."
	usage
	exit 1
fi

clear_outfile

case $1 in
	--apache|-a)
		check_baculum_cfgs
		check_apache_cfgs
		check_baculum_apache_files
		;;
	--lighttpd|-l)
		check_baculum_cfgs
		check_lighttpd_cfgs
		check_baculum_lighttpd_files
		;;
	--version|-v)
		version
		;;
	--help|-h)
		usage
		;;
	*)
		usage
		;;
esac

exit $ERROR
