/*
   Bacula(R) - The Network Backup Solution

   Copyright (C) 2000-2023 Kern Sibbald

   The original author of Bacula is Kern Sibbald, with contributions
   from many others, a complete list can be found in the file AUTHORS.

   You may use this file and others of this release according to the
   license defined in the LICENSE file, which includes the Affero General
   Public License, v3.0 ("AGPLv3") and some additional permissions and
   terms pursuant to its AGPLv3 Section 7.

   This notice must be preserved when any source code is
   conveyed and/or propagated.

   Bacula(R) is a registered trademark of Kern Sibbald.
*/
/**
 * Main program to test loading and running Bacula plugins.
 *   Destined to become Bacula pluginloader, ...
 *
 * Kern Sibbald, October 2007
 */
#include "bacula.h"
#include "filed.h"

extern CLIENT *me;
extern DLL_IMP_EXP char *exepath;
extern DLL_IMP_EXP char *sysconfigpath;
extern DLL_IMP_EXP char *version;
extern DLL_IMP_EXP char *dist_name;
extern DLL_IMP_EXP int beef;

const int dbglvl  = 55;
const int dbglvl2 = 150;
#ifdef HAVE_WIN32
const char *plugin_type = "-fd.dll";
#else
const char *plugin_type = "-fd.so";
#endif

extern int save_file(JCR *jcr, FF_PKT *ff_pkt, bool top_level);
extern bool check_changes(JCR *jcr, FF_PKT *ff_pkt);
extern bool metadata_save(JCR *jcr, const plugin_metadata *plug_meta);

/* Function pointers to be set here */
extern DLL_IMP_EXP int     (*plugin_bopen)(BFILE *bfd, const char *fname, uint64_t flags, mode_t mode);
extern DLL_IMP_EXP int     (*plugin_bclose)(BFILE *bfd);
extern DLL_IMP_EXP ssize_t (*plugin_bread)(BFILE *bfd, void *buf, size_t count);
extern DLL_IMP_EXP ssize_t (*plugin_bwrite)(BFILE *bfd, void *buf, size_t count);
extern DLL_IMP_EXP boffset_t (*plugin_blseek)(BFILE *bfd, boffset_t offset, int whence);


/* Forward referenced functions */
static bRC baculaGetValue(bpContext *ctx, bVariable var, void *value);
static bRC baculaSetValue(bpContext *ctx, bVariable var, void *value);
static bRC baculaRegisterEvents(bpContext *ctx, ...);
static bRC baculaJobMsg(bpContext *ctx, const char *file, int line,
  int type, utime_t mtime, const char *fmt, ...);
static bRC baculaDebugMsg(bpContext *ctx, const char *file, int line,
  int level, const char *fmt, ...);
static void *baculaMalloc(bpContext *ctx, const char *file, int line,
              size_t size);
static void baculaFree(bpContext *ctx, const char *file, int line, void *mem);
static bRC  baculaAddExclude(bpContext *ctx, const char *file);
static bRC baculaAddInclude(bpContext *ctx, const char *file);
static bRC baculaAddOptions(bpContext *ctx, const char *opts);
static bRC baculaAddRegex(bpContext *ctx, const char *item, int type);
static bRC baculaAddWild(bpContext *ctx, const char *item, int type);
static bRC baculaNewOptions(bpContext *ctx);
static bRC baculaNewInclude(bpContext *ctx);
static bRC baculaNewPreInclude(bpContext *ctx);
static bool is_plugin_compatible(Plugin *plugin);
static bool get_plugin_name(JCR *jcr, char *cmd, int *ret);
static bRC baculaCheckChanges(bpContext *ctx, struct save_pkt *sp);
static bRC baculaAcceptFile(bpContext *ctx, struct save_pkt *sp);
static bRC baculaAccurateAttribs(bpContext *ctx, accurate_attribs_pkt *att);
static void plugin_register_verify_data(bpContext *ctx);
static bRC baculaAddPlugin(bpContext *ctx, const char *file);
static bRC baculaAddFileEvent(bpContext *ctx, struct fileevent_pkt *event);

/*
 * These will be plugged into the global pointer structure for
 *  the findlib.
 */
static int     my_plugin_bopen(BFILE *bfd, const char *fname, uint64_t flags, mode_t mode);
static int     my_plugin_bclose(BFILE *bfd);
static ssize_t my_plugin_bread(BFILE *bfd, void *buf, size_t count);
static ssize_t my_plugin_bwrite(BFILE *bfd, void *buf, size_t count);
static boffset_t my_plugin_blseek(BFILE *bfd, boffset_t offset, int whence);


/* Bacula info */
static bInfo binfo = {
   sizeof(bInfo),
   FD_PLUGIN_INTERFACE_VERSION
};

/* Bacula entry points when calling back from the plugin */
static bFuncs bfuncs = {
   sizeof(bFuncs),
   FD_PLUGIN_INTERFACE_VERSION,
   baculaRegisterEvents,
   baculaGetValue,
   baculaSetValue,
   baculaJobMsg,
   baculaDebugMsg,
   baculaMalloc,
   baculaFree,
   baculaAddExclude,
   baculaAddInclude,
   baculaAddOptions,
   baculaAddRegex,
   baculaAddWild,
   baculaNewOptions,
   baculaNewInclude,
   baculaNewPreInclude,
   baculaCheckChanges,
   baculaAcceptFile,
   baculaAccurateAttribs,
   baculaAddPlugin,
   baculaAddFileEvent
};

/*
 * Bacula private context
 */
struct bacula_ctx {
   JCR *jcr;                             /* jcr for plugin */
   bRC  rc;                              /* last return code */
   bool disabled;                        /* set if plugin disabled */
   bool restoreFileStarted;
   bool createFileCalled;
   bool verifyFileCalled;                /* true if startVerifyFile() has been called */
   bool cancelCalled;                    /* true if the plugin got the cancel event */
   findINCEXE *exclude;                  /* pointer to exclude files */
   findINCEXE *include;                  /* pointer to include/exclude files */
   Plugin *plugin;                       /* pointer to the plugin itself */
};

static bacula_ctx *get_bacula_ctx(bpContext *plugin_ctx)
{
   if (plugin_ctx) {
      return (bacula_ctx *) plugin_ctx->bContext;
   }
   return NULL;
}

/*
 * Test if event is for this plugin
 */
static bool for_this_plugin(Plugin *plugin, char *name, int len)
{
   Dmsg4(dbglvl, "name=%s len=%d plugin=%s plen=%d\n", name, len, plugin->file, plugin->file_len);
   if (!name) {   /* if no plugin name, all plugins get it */
      return true;
   }
   /* Return global VSS job metadata to all plugins */
   if (strcmp("job", name) == 0) {  /* old V4.0 name for VSS job metadata */
      return true;
   }
   if (strcmp("*all*", name) == 0) { /* new v6.0 name for VSS job metadata */
      return true;
   }
   /* Check if this is the correct plugin */
   if (len == plugin->file_len && strncmp(plugin->file, name, len) == 0) {
      return true;
   }
   return false;
}


bool is_plugin_disabled(bpContext *plugin_ctx)
{
   bacula_ctx *b_ctx;
   Dsm_check(999);
   if (!plugin_ctx) {
      return true;
   }
   b_ctx = (bacula_ctx *)plugin_ctx->bContext;
   if (!b_ctx) {
      return true;
   }
   return b_ctx->disabled;
}

bool is_plugin_disabled(JCR *jcr)
{
   return is_plugin_disabled(jcr->plugin_ctx);
}

/**
 * Create a plugin event When receiving bEventCancelCommand, this function is
 * called by an other thread.
 */
void generate_plugin_event(JCR *jcr, bEventType eventType, void *value)
{
   bpContext *plugin_ctx;
   bacula_ctx *bac_ctx;
   bEvent event;
   Plugin *plugin;
   char *name = NULL;
   int i;
   int len = 0;
   bool call_if_canceled = false;
   restore_object_pkt *rop;

   Dsm_check(999);
   if (!b_plugin_list || !jcr || !jcr->plugin_ctx_list) {
      return;                         /* Return if no plugins loaded */
   }

   /*
    * Some events are sent to only a particular plugin or must be
    *  called even if the job is canceled
    */
   switch(eventType) {
   case bEventPluginCommand:
   case bEventPluginOptions:
   case bEventOptionPlugin:
      name = (char *)value;
      if (!get_plugin_name(jcr, name, &len)) {
         return;
      }
      break;
   case bEventRestoreObject:
      /* After all RestoreObject, we have it one more time with value=NULL */
      if (value) {
         /* Some RestoreObjects may not have a plugin name */
         rop = (restore_object_pkt *)value;
         if (*rop->plugin_name) {
            name = rop->plugin_name;
            get_plugin_name(jcr, name, &len);
         }
      }

      break;
   case bEventEndBackupJob:
   case bEventEndVerifyJob:
      call_if_canceled = true; /* plugin *must* see this call */
      break;
   case bEventStartVerifyJob:
   case bEventStartRestoreJob:
      break;
   case bEventEndRestoreJob:
      call_if_canceled = true; /* plugin *must* see this call */
      break;
   case bEventCancelCommand:
      call_if_canceled = true;  /* plugin *must* see this call */
      break;
   default:
      break;
   }

   /* If call_if_canceled is set, we call the plugin anyway */
   if (!call_if_canceled && jcr->is_job_canceled()) {
      return;
   }

   bpContext *plugin_ctx_list = (bpContext *)jcr->plugin_ctx_list;
   event.eventType = eventType;

   Dmsg2(dbglvl, "plugin_ctx=%p JobId=%d\n", jcr->plugin_ctx_list, jcr->JobId);

   /*
    * Pass event to every plugin (except if name is set). If name
    *   is set, we pass it only to the plugin with that name.
    */
   foreach_alist_index(i, plugin, b_plugin_list) {
      if (!for_this_plugin(plugin, name, len)) {
         Dmsg2(dbglvl, "Not for this plugin name=%s NULL=%d\n",
            name, name==NULL?1:0);
         continue;
      }
      /*
       * Note, at this point do not change
       *   jcr->plugin or jcr->plugin_ctx
       */
      Dsm_check(999);
      plugin_ctx = &plugin_ctx_list[i];
      bac_ctx = get_bacula_ctx(plugin_ctx);

      if (is_plugin_disabled(plugin_ctx)) {
         Dmsg1(50, "Plugin %s disabled\n", plugin->file);
         continue;
      }
      if (eventType == bEventCancelCommand) {
         if (bac_ctx->cancelCalled) {
            continue;
         }
         bac_ctx->cancelCalled = true;
      }
      if (eventType == bEventEndRestoreJob) {
         Dmsg0(50, "eventType==bEventEndRestoreJob\n");
         if (bac_ctx && bac_ctx->restoreFileStarted) {
            plug_func(jcr->plugin)->endRestoreFile(jcr->plugin_ctx);
         }
         if (bac_ctx) {
            bac_ctx->restoreFileStarted = false;
            bac_ctx->createFileCalled = false;
         }
      }
      plug_func(plugin)->handlePluginEvent(plugin_ctx, &event, value);
   }
   return;
}

/* Return the plugin index in our list, can be used to call
 * massively a plugin like in plugin_send_restorefilelist().
 * If the plugin is not here or disabled, we return -1
 */
int plugin_get_idx(JCR *jcr, char *fname)
{
   int i;
   Plugin *plugin;
   bpContext *plugin_ctx_list;
   if (!b_plugin_list || !jcr || !jcr->plugin_ctx_list || jcr->is_job_canceled()) {
      return -1;                      /* Return if no plugins loaded */
   }

   plugin_ctx_list = (bpContext *)jcr->plugin_ctx_list;

   /* Pass event to every plugin */
   foreach_alist_index(i, plugin, b_plugin_list) {
      if (strcmp(plugin->file, fname) == 0) {
         jcr->plugin_ctx = &plugin_ctx_list[i];
         jcr->plugin = (Plugin*)b_plugin_list->get(i);

         if (is_plugin_disabled(jcr)) {
            i = -1;             /* not usable */
         }

         jcr->plugin_ctx = NULL;
         jcr->plugin = NULL;
         return i;
      }
   }
   return -1;                   /* not found */
}

/* Need to get the plugin index first (with plugin_get_idx()), then
 * the function is called for each file that we get.
 *
 * The plugin informs us that it is interested with the file list
 * in the pluginfeatures_cmd() DIR -> FD commamnd.
 *
 * The plugin must define the restoreFileList handler and return 
 * PLUGIN_FEATURE_RESTORELISTFILES in the handlePluginEvent(bEventFeatures).
 */
bool plugin_send_restorefilelist(JCR *jcr, int plugin_index,
                                 char *path,
                                 char *lstat, /* Can be decoded with decode_stat() */
                                 char *checksum, /* if available */
                                 int delta_seq)
{
   bool ret=false;
   bRC rc;
   bpContext *plugin_ctx_list;
   struct restore_filelist_pkt rp;

   if (plugin_index == -1 || !b_plugin_list || !jcr ||
       !jcr->plugin_ctx_list || jcr->is_job_canceled())
   {
      goto bail_out;
   }

   plugin_ctx_list = (bpContext *)jcr->plugin_ctx_list;
   jcr->plugin_ctx = &plugin_ctx_list[plugin_index];
   jcr->plugin = (Plugin*)b_plugin_list->get(plugin_index);

   if (plug_func(jcr->plugin)->restoreFileList == NULL) {
      goto bail_out;
   }
   if (path == NULL) {
      rc = plug_func(jcr->plugin)->restoreFileList(jcr->plugin_ctx, NULL);

   } else {
      decode_stat(lstat, &rp.statp, sizeof(rp.statp), &rp.LinkFI);
      rp.pkt_size = sizeof(rp);
      rp.pkt_end = sizeof(rp);
      rp.delta_seq = delta_seq;
      rp.file_index = 0;
      rp.attrEx = "";
      rp.ofname = path;
      rp.chksum = checksum;
      rc = plug_func(jcr->plugin)->restoreFileList(jcr->plugin_ctx, &rp);
   }

   if (rc == bRC_OK) {
      ret = true;
   }

bail_out:
   jcr->plugin_ctx = NULL;
   jcr->plugin = NULL;
   return ret;
}


/**
 * Check if file was seen for accurate
 */
bool plugin_check_file(JCR *jcr, char *fname)
{
   Plugin *plugin;
   int rc = bRC_OK;
   int i;

   Dsm_check(999);
   if (!b_plugin_list || !jcr || !jcr->plugin_ctx_list || jcr->is_job_canceled()) {
      return false;                      /* Return if no plugins loaded */
   }

   bpContext *plugin_ctx_list = (bpContext *)jcr->plugin_ctx_list;

   Dmsg2(dbglvl, "plugin_ctx=%p JobId=%d\n", jcr->plugin_ctx_list, jcr->JobId);

   /* Pass event to every plugin */
   foreach_alist_index(i, plugin, b_plugin_list) {
      jcr->plugin_ctx = &plugin_ctx_list[i];
      jcr->plugin = plugin;
      if (is_plugin_disabled(jcr)) {
         continue;
      }
      if (plug_func(plugin)->checkFile == NULL) {
         continue;
      }
      rc = plug_func(plugin)->checkFile(jcr->plugin_ctx, fname);
      if (rc == bRC_Seen) {
         break;
      }
   }

   Dsm_check(999);
   jcr->plugin = NULL;
   jcr->plugin_ctx = NULL;
   return rc == bRC_Seen;
}

/* Get the first part of the the plugin command
 *  systemstate:/@SYSTEMSTATE/
 * => ret = 11
 * => can use for_this_plugin(plug, cmd, ret);
 *
 * The plugin command can contain only the plugin name
 *  Plugin = alldrives
 * => ret = 9
 */
static bool get_plugin_name(JCR *jcr, char *cmd, int *ret)
{
   char *p;
   int len;
   Dsm_check(999);
   if (!cmd || (*cmd == '\0')) {
      return false;
   }
   /* Handle plugin command here backup */
   Dmsg1(dbglvl, "plugin cmd=%s\n", cmd);
   if ((p = strchr(cmd, ':')) == NULL) {
      if (strchr(cmd, ' ') == NULL) { /* we have just the plugin name */
         len = strlen(cmd);
      } else {
         Jmsg1(jcr, M_ERROR, 0, "Malformed plugin command: %s\n", cmd);
         return false;
      }
   } else {                     /* plugin:argument */
      len = p - cmd;
      if (len <= 0) {
         return false;
      }
   }
   *ret = len;
   Dsm_check(999);
   return true;
}


static void update_ff_pkt(FF_PKT *ff_pkt, struct save_pkt *sp)
{
   Dsm_check(999);
   ff_pkt->no_read = sp->no_read;
   ff_pkt->delta_seq = sp->delta_seq;

   if (sp->flags & FO_DELTA) {
      ff_pkt->flags |= FO_DELTA;
      ff_pkt->delta_seq++;          /* make new delta sequence number */
   } else {
      ff_pkt->flags &= ~FO_DELTA;   /* clean delta sequence number */
      ff_pkt->delta_seq = 0;
   }

   if (sp->flags & FO_OFFSETS) {
      ff_pkt->flags |= FO_OFFSETS;
   } else {
      ff_pkt->flags &= ~FO_OFFSETS;
   }
   /* Sparse code doesn't work with plugins
    * that use FIFO or STDOUT/IN to communicate
    */
   if (sp->flags & FO_SPARSE) {
      ff_pkt->flags |= FO_SPARSE;
   } else {
      ff_pkt->flags &= ~FO_SPARSE;
   }
   if (sp->flags & FO_PORTABLE) {
      ff_pkt->flags |= FO_PORTABLE;
   } else {
      ff_pkt->flags &= ~FO_PORTABLE;
   }
   ff_pkt->flags |= FO_PLUGIN;       /* data from plugin */
   Dsm_check(999);
}

/* Look if a given plugin can be used */
bool is_plugin_loaded(JCR *jcr, char *cmd)
{
   Plugin *plugin;
   bool found=false;
   int len;
   int i=0;

   bpContext *plugin_ctx_list = (bpContext *)jcr->plugin_ctx_list;
   if (!b_plugin_list || !jcr->plugin_ctx_list || jcr->is_job_canceled()) {
      goto bail_out;         /* Return if no plugins loaded */
   }

   if (!get_plugin_name(jcr, cmd, &len)) {
      goto bail_out;
   }

   /* Note, we stop the loop on the first plugin that matches the name */
   foreach_alist_index(i, plugin, b_plugin_list) {
      Dmsg4(dbglvl, "plugin=%s plen=%d cmd=%s len=%d\n", plugin->file, plugin->file_len, cmd, len);
      if (!for_this_plugin(plugin, cmd, len)) {
         continue;
      }
      if (is_plugin_disabled(&plugin_ctx_list[i])) {
         goto bail_out;
      }
      found=true;
      goto bail_out;
   } /* end foreach loop */
bail_out:
   return found;
}

/* Ask to a Option Plugin what to do with the current file */
bRC plugin_option_handle_file(JCR *jcr, FF_PKT *ff_pkt, struct save_pkt *sp)
{
   Plugin *plugin;
   bRC ret = bRC_Error;
   bool found=false;
   char *cmd = ff_pkt->plugin;
   int len;
   int i=0;
   bEvent event;
   event.eventType = bEventHandleBackupFile;

   Dsm_check(999);
   bpContext *plugin_ctx_list = (bpContext *)jcr->plugin_ctx_list;
   memset(sp, 0, sizeof(struct save_pkt));
   sp->pkt_size = sp->pkt_end = sizeof(struct save_pkt);
   sp->portable = true;
   sp->cmd = cmd;
   sp->link = ff_pkt->link;
   sp->cmd = ff_pkt->plugin;
   sp->statp = ff_pkt->statp;
   sp->fname = ff_pkt->fname;
   sp->snap_fname = ff_pkt->snap_fname;
   sp->delta_seq = ff_pkt->delta_seq;
   sp->accurate_found = ff_pkt->accurate_found;

   /* We can handle only regular files */
   switch (ff_pkt->type) {
   case FT_REGE:
   case FT_REG:
      break;
   default:
      return bRC_Core;
   }
   
   if (!b_plugin_list || !jcr->plugin_ctx_list || jcr->is_job_canceled()) {
      Jmsg1(jcr, M_FATAL, 0, "Command plugin \"%s\" requested, but is not loaded.\n", cmd);
      goto bail_out;         /* Return if no plugins loaded */
   }

   if (!get_plugin_name(jcr, cmd, &len)) {
      goto bail_out;
   }

   /* Note, we stop the loop on the first plugin that matches the name */
   foreach_alist_index(i, plugin, b_plugin_list) {
      Dmsg4(dbglvl, "plugin=%s plen=%d cmd=%s len=%d\n", plugin->file, plugin->file_len, cmd, len);
      if (!for_this_plugin(plugin, cmd, len)) {
         continue;
      }

      found=true;

      Dsm_check(999);
      if (is_plugin_disabled(&plugin_ctx_list[i])) {
         goto bail_out;
      }

      jcr->plugin_ctx = &plugin_ctx_list[i];
      jcr->plugin = plugin;

      ret = plug_func(plugin)->handlePluginEvent(&plugin_ctx_list[i],
                                                 &event, sp);

      /* TODO: would be better to set this in save_file() */
      if (ret == bRC_OK) {
         jcr->opt_plugin = true;
         jcr->plugin = plugin;
         jcr->plugin_sp = sp;      /* Unset sp in save_file */
         jcr->plugin_ctx = &plugin_ctx_list[i];

         update_ff_pkt(ff_pkt, sp);
      /* reset plugin in JCR if not used this time */
      } else {
         jcr->plugin_ctx = NULL;
         jcr->plugin = NULL;
      }

      goto bail_out;
   } /* end foreach loop */
bail_out:
   if (!found) {
      Jmsg1(jcr, M_FATAL, 0, "Options plugin \"%s\" not found.\n", cmd);
   }
   Dsm_check(999);
   return ret;
}

/**
 * Sequence of calls for a backup:
 * 1. plugin_save() here is called with ff_pkt
 * 2. we find the plugin requested on the command string
 * 3. we generate a bEventBackupCommand event to the specified plugin
 *    and pass it the command string.
 * 4. we make a startPluginBackup call to the plugin, which gives
 *    us the data we need in save_pkt
 * 5. we call Bacula's save_file() subroutine to save the specified
 *    file.  The plugin will be called at pluginIO() to supply the
 *    file data.
 *
 * Sequence of calls for restore:
 *   See subroutine plugin_name_stream() below.
 */
int plugin_save(JCR *jcr, FF_PKT *ff_pkt, bool top_level)
{
   Plugin *plugin;
   int len;
   int i;
   char *cmd = ff_pkt->top_fname;
   struct save_pkt sp;
   bEvent event;
   POOL_MEM fname(PM_FNAME);
   POOL_MEM link(PM_FNAME);

   Dsm_check(999);
   if (!b_plugin_list || !jcr->plugin_ctx_list || jcr->is_job_canceled()) {
      Jmsg1(jcr, M_FATAL, 0, "Command plugin \"%s\" requested, but is not loaded.\n", cmd);
      return 1;                            /* Return if no plugins loaded */
   }

   jcr->cmd_plugin = true;
   bpContext *plugin_ctx_list = (bpContext *)jcr->plugin_ctx_list;
   event.eventType = bEventBackupCommand;

   if (!get_plugin_name(jcr, cmd, &len)) {
      goto bail_out;
   }

   /* Note, we stop the loop on the first plugin that matches the name */
   foreach_alist_index(i, plugin, b_plugin_list) {
      Dmsg4(dbglvl, "plugin=%s plen=%d cmd=%s len=%d\n", plugin->file, plugin->file_len, cmd, len);
      if (!for_this_plugin(plugin, cmd, len)) {
         continue;
      }
      /*
       * We put the current plugin pointer, and the plugin context
       *  into the jcr, because during save_file(), the plugin
       *  will be called many times and these values are needed.
       */
      Dsm_check(999);
      jcr->plugin_ctx = &plugin_ctx_list[i];
      jcr->plugin = plugin;
      if (is_plugin_disabled(jcr)) {
         goto bail_out;
      }
      if (!plug_func(plugin)->startBackupFile || !plug_func(plugin)->endBackupFile) {
         Dmsg1(dbglvl, "Plugin not suitable for Backup job %s\n", cmd);
         goto bail_out;
      }
      Dmsg1(dbglvl, "Command plugin = %s\n", cmd);
      /* Send the backup command to the right plugin*/
      if (plug_func(plugin)->handlePluginEvent(jcr->plugin_ctx, &event, cmd) != bRC_OK) {
         goto bail_out;
      }
      /* Loop getting filenames to backup then saving them */
      while (!jcr->is_job_canceled()) {
         memset(&sp, 0, sizeof(sp));
         sp.pkt_size = sizeof(sp);
         sp.pkt_end = sizeof(sp);
         sp.do_dedup = is_dedup_enabled(jcr, jcr->ff);
         sp.portable = true;
         sp.no_read = false;
         sp.flags = 0;
         sp.cmd = cmd;
         Dmsg3(dbglvl, "startBackup st_size=%p st_blocks=%p sp=%p\n", &sp.statp.st_size, &sp.statp.st_blocks,
               &sp);
         Dsm_check(999);
         /* Get the file save parameters. I.e. the stat pkt ... */
         if (plug_func(plugin)->startBackupFile(jcr->plugin_ctx, &sp) != bRC_OK) {
            goto bail_out;
         }

         /* Set the flag if the plugin wants to update file's attributues */
         ff_pkt->stat_update = sp.stat_update;

         if (sp.type == 0) {
            Jmsg1(jcr, M_FATAL, 0, _("Command plugin \"%s\": no type in startBackupFile packet.\n"),
               cmd);
            goto bail_out;
         }
         jcr->plugin_sp = &sp;
         ff_pkt = jcr->ff;
         /*
          * Copy fname and link because save_file() zaps them.  This
          *  avoids zaping the plugin's strings.
          */
         ff_pkt->type = sp.type;
         if (IS_FT_OBJECT(sp.type)) {
            if (sp.type != FT_PLUGIN_OBJECT && !sp.restore_obj.object_name) {
               Jmsg1(jcr, M_FATAL, 0, _("Command plugin \"%s\": no object_name in startBackupFile packet.\n"),
                  cmd);
               goto bail_out;
            }
            ff_pkt->fname = cmd;                 /* full plugin string */
            if (sp.type == FT_PLUGIN_OBJECT) {
               ff_pkt->plugin_obj.path = sp.plugin_obj.path;
               ff_pkt->plugin_obj.plugin_name = cmd;
               ff_pkt->plugin_obj.object_category = sp.plugin_obj.object_category;
               ff_pkt->plugin_obj.object_type = sp.plugin_obj.object_type;
               ff_pkt->plugin_obj.object_name = sp.plugin_obj.object_name;
               ff_pkt->plugin_obj.object_source = sp.plugin_obj.object_source;
               ff_pkt->plugin_obj.object_uuid = sp.plugin_obj.object_uuid;
               ff_pkt->plugin_obj.object_size = sp.plugin_obj.object_size;
               ff_pkt->plugin_obj.status = sp.plugin_obj.status;
               ff_pkt->plugin_obj.count = sp.plugin_obj.count;
            } else {
               ff_pkt->restore_obj.object_name = sp.restore_obj.object_name;
               ff_pkt->restore_obj.index = sp.restore_obj.index;     /* restore object index */
               ff_pkt->restore_obj.object_compression = 0;      /* no compression for now */
               ff_pkt->restore_obj.object = sp.restore_obj.object;
               ff_pkt->restore_obj.object_len = sp.restore_obj.object_len;
            }
         } else {
            Dsm_check(999);
            if (!sp.fname) {
               Jmsg1(jcr, M_FATAL, 0, _("Command plugin \"%s\": no fname in startBackupFile packet.\n"),
                  cmd);
               goto bail_out;
            }
            pm_strcpy(fname, sp.fname);
            pm_strcpy(link, sp.link);

            ff_pkt->plug_meta = NULL;
            if (sp.plug_meta){
               /* File has some metadata assigned to it */
               ff_pkt->plug_meta = sp.plug_meta;
            }

            ff_pkt->fname = fname.c_str();
            ff_pkt->snap_fname = fname.c_str();
            ff_pkt->link = link.c_str();
            ff_pkt->LinkFI = sp.LinkFI;
            update_ff_pkt(ff_pkt, &sp);
         }

         memcpy(&ff_pkt->statp, &sp.statp, sizeof(ff_pkt->statp));
         Dmsg2(dbglvl, "startBackup returned type=%d, fname=%s\n", sp.type, sp.fname);
         if (sp.restore_obj.object) {
            Dmsg2(dbglvl, "index=%d object=%s\n", sp.restore_obj.index, sp.restore_obj.object);
         }
         /* Call Bacula core code to backup the plugin's file */
         save_file(jcr, ff_pkt, true);
         bRC rc = plug_func(plugin)->endBackupFile(jcr->plugin_ctx);
         if (rc == bRC_More || rc == bRC_OK) {
            accurate_mark_file_as_seen(jcr, fname.c_str());
         }
         Dsm_check(999);
         if (rc == bRC_More) {
            continue;
         }
         goto bail_out;
      } /* end while loop */
      goto bail_out;
   } /* end loop over all plugins */
   Jmsg1(jcr, M_FATAL, 0, "Command plugin \"%s\" not found.\n", cmd);

bail_out:
   Dsm_check(999);
   jcr->cmd_plugin = false;
   jcr->plugin = NULL;
   jcr->plugin_ctx = NULL;
   return 1;
}


/**
 * Sequence of calls for a estimate:
 * 1. plugin_estimate() here is called with ff_pkt
 * 2. we find the plugin requested on the command string
 * 3. we generate a bEventEstimateCommand event to the specified plugin
 *    and pass it the command string.
 * 4. we make a startPluginBackup call to the plugin, which gives
 *    us the data we need in save_pkt
 *
 */
int plugin_estimate(JCR *jcr, FF_PKT *ff_pkt, bool top_level)
{
   Plugin *plugin;
   int len;
   int i;
   char *cmd = ff_pkt->top_fname;
   struct save_pkt sp;
   bEvent event;
   POOL_MEM fname(PM_FNAME);
   POOL_MEM link(PM_FNAME);
   ATTR attr;

   Dsm_check(999);
   if (!b_plugin_list || !jcr->plugin_ctx_list) {
      Jmsg1(jcr, M_FATAL, 0, "Command plugin \"%s\" requested, but is not loaded.\n", cmd);
      return 1;                            /* Return if no plugins loaded */
   }

   jcr->cmd_plugin = true;
   bpContext *plugin_ctx_list = (bpContext *)jcr->plugin_ctx_list;
   event.eventType = bEventEstimateCommand;

   if (!get_plugin_name(jcr, cmd, &len)) {
      goto bail_out;
   }

   /* Note, we stop the loop on the first plugin that matches the name */
   foreach_alist_index(i, plugin, b_plugin_list) {
      Dmsg4(dbglvl, "plugin=%s plen=%d cmd=%s len=%d\n", plugin->file, plugin->file_len, cmd, len);
      if (!for_this_plugin(plugin, cmd, len)) {
         continue;
      }
      /*
       * We put the current plugin pointer, and the plugin context
       *  into the jcr, because during save_file(), the plugin
       *  will be called many times and these values are needed.
       */
      Dsm_check(999);
      jcr->plugin_ctx = &plugin_ctx_list[i];
      jcr->plugin = plugin;
      if (is_plugin_disabled(jcr)) {
         goto bail_out;
      }
      if (!plug_func(plugin)->startBackupFile || !plug_func(plugin)->endBackupFile) {
         Dmsg1(dbglvl, "Plugin not suitable for Estimate job %s\n", cmd);
         goto bail_out;
      }
      Dmsg1(dbglvl, "Command plugin = %s\n", cmd);
      /* Send the backup command to the right plugin*/
      if (plug_func(plugin)->handlePluginEvent(jcr->plugin_ctx, &event, cmd) != bRC_OK) {
         goto bail_out;
      }
      /* Loop getting filenames to backup then saving them */
      while (!jcr->is_job_canceled()) {
         Dsm_check(999);
         memset(&sp, 0, sizeof(sp));
         sp.pkt_size = sizeof(sp);
         sp.pkt_end = sizeof(sp);
         sp.portable = true;
         sp.flags = 0;
         sp.cmd = cmd;
         Dmsg3(dbglvl, "startBackup st_size=%p st_blocks=%p sp=%p\n", &sp.statp.st_size, &sp.statp.st_blocks,
                &sp);
         /* Get the file save parameters. I.e. the stat pkt ... */
         if (plug_func(plugin)->startBackupFile(jcr->plugin_ctx, &sp) != bRC_OK) {
            goto bail_out;
         }
         if (sp.type == 0) {
            Jmsg1(jcr, M_FATAL, 0, _("Command plugin \"%s\": no type in startBackupFile packet.\n"),
               cmd);
            goto bail_out;
         }

         if (!IS_FT_OBJECT(sp.type)) {
            if (!sp.fname) {
               Jmsg1(jcr, M_FATAL, 0, _("Command plugin \"%s\": no fname in startBackupFile packet.\n"),
                  cmd);
               goto bail_out;
            }

            /* Count only files backed up */
            switch (sp.type) {
            case FT_REGE:
            case FT_REG:
            case FT_LNK:
            case FT_DIREND:
            case FT_SPEC:
            case FT_RAW:
            case FT_FIFO:
            case FT_LNKSAVED:
               jcr->JobFiles++;        /* increment number of files backed up */
               break;
            default:
               break;
            }
            jcr->num_files_examined++;

            /* we expect that plugin could generate any type of backup file so we should count its size too */
            if (sp.type != FT_LNKSAVED) {
               if (sp.statp.st_size > 0) {
                  jcr->JobBytes += sp.statp.st_size;
               }
            }

            if (jcr->listing) {
               memcpy(&attr.statp, &sp.statp, sizeof(struct stat));
               attr.type = sp.type;
               attr.ofname = (POOLMEM *)sp.fname;
               attr.olname = (POOLMEM *)sp.link;
               print_ls_output(jcr, &attr);
            }
         }

         Dmsg2(dbglvl, "startBackup returned type=%d, fname=%s\n", sp.type, sp.fname);
         if (sp.restore_obj.object) {
            Dmsg2(dbglvl, "index=%d object=%s\n", sp.restore_obj.index, sp.restore_obj.object);
         }
         bRC rc = plug_func(plugin)->endBackupFile(jcr->plugin_ctx);
         if (rc == bRC_More || rc == bRC_OK) {
            accurate_mark_file_as_seen(jcr, sp.fname);
         }
         Dsm_check(999);
         if (rc == bRC_More) {
            continue;
         }
         goto bail_out;
      } /* end while loop */
      goto bail_out;
   } /* end loop over all plugins */
   Jmsg1(jcr, M_FATAL, 0, "Command plugin \"%s\" not found.\n", cmd);

bail_out:
   Dsm_check(999);
   jcr->cmd_plugin = false;
   jcr->plugin = NULL;
   jcr->plugin_ctx = NULL;
   return 1;
}

/*
 * Send plugin name start/end record to SD
 */
bool send_plugin_name(JCR *jcr, BSOCK *sd, bool start)
{
   int stat;
   int index = jcr->JobFiles;
   struct save_pkt *sp = (struct save_pkt *)jcr->plugin_sp;
   int32_t stream = STREAM_PLUGIN_NAME;

   Dsm_check(999);
   if (!sp) {
      Jmsg0(jcr, M_FATAL, 0, _("Plugin save packet not found.\n"));
      return false;
   }
   if (jcr->is_job_canceled()) {
      return false;
   }

   if (start) {
      index++;                  /* JobFiles not incremented yet */
   }
   Dmsg1(dbglvl, "send_plugin_name=%s\n", sp->cmd);
   /* Send stream header */
   Dsm_check(999);
   if (start) {
      plugin_check_stream(jcr, stream);    /* get stream modified by plugin */
   }
   if (!sd->fsend("%ld %d 0", index, stream)) {
      Jmsg1(jcr, M_FATAL, 0, _("Network send error to SD. ERR=%s\n"),
            sd->bstrerror());
      return false;
   }
   Dmsg1(dbglvl, "send plugin name hdr: %s\n", sd->msg);

   Dsm_check(999);
   if (start) {
      /* Send data -- not much */
      stat = sd->fsend("%ld 1 %d %s%c", index, sp->portable, sp->cmd, 0);
   } else {
      /* Send end of data */
      stat = sd->fsend("%ld 0", jcr->JobFiles);
   }
   Dsm_check(999);
   if (!stat) {
      Jmsg1(jcr, M_FATAL, 0, _("Network send error to SD. ERR=%s\n"),
            sd->bstrerror());
         return false;
   }
   Dmsg1(dbglvl, "send plugin start/end: %s\n", sd->msg);
   sd->signal(BNET_EOD);            /* indicate end of plugin name data */
   Dsm_check(999);
   return true;
}

/**
 * Plugin name stream found during restore.  The record passed in
 *  argument name was generated in send_plugin_name() above.
 *
 * Returns: true  if start of stream
 *          false if end of steam
 */
bool plugin_name_stream(JCR *jcr, char *name)
{
   char *p = name;
   char *cmd;
   bool start;
   Plugin *plugin;
   int len;
   int i;
   bpContext *plugin_ctx_list = jcr->plugin_ctx_list;
   bacula_ctx *bac_ctx = get_bacula_ctx(jcr->plugin_ctx); /* current context */

   Dsm_check(999);
   Dmsg1(dbglvl, "Read plugin stream string=%s\n", name);
   skip_nonspaces(&p);             /* skip over jcr->JobFiles */
   skip_spaces(&p);
   start = *p == '1';
   if (start) {
      /* Start of plugin data */
      skip_nonspaces(&p);          /* skip start/end flag */
      skip_spaces(&p);
//    portable = *p == '1';
      skip_nonspaces(&p);          /* skip portable flag */
      skip_spaces(&p);
      cmd = p;
   } else {
      /*
       * End of plugin data, notify plugin, then clear flags
       */
      Dmsg2(dbglvl, "End plugin data plugin=%p ctx=%p\n", jcr->plugin, jcr->plugin_ctx);
      if (bac_ctx && bac_ctx->restoreFileStarted) {
         plug_func(jcr->plugin)->endRestoreFile(jcr->plugin_ctx);
      }
      if (bac_ctx) {
         bac_ctx->restoreFileStarted = false;
         bac_ctx->createFileCalled = false;
      }
      jcr->plugin_ctx = NULL;
      jcr->plugin = NULL;
      goto bail_out;
   }
   Dsm_check(999);
   if (!plugin_ctx_list) {
      goto bail_out;
   }

   /*
    * After this point, we are dealing with a restore start
    */
   if (!get_plugin_name(jcr, cmd, &len)) {
      goto bail_out;
   }

   /*
    * Search for correct plugin as specified on the command
    */
   Dsm_check(999);
   foreach_alist_index(i, plugin, b_plugin_list) {
      bEvent event;
      Dmsg3(dbglvl, "plugin=%s cmd=%s len=%d\n", plugin->file, cmd, len);
      if (!for_this_plugin(plugin, cmd, len)) {
         continue;
      }
      Dsm_check(999);
      jcr->plugin_ctx = &plugin_ctx_list[i];
      jcr->plugin = plugin;
      bac_ctx = get_bacula_ctx(jcr->plugin_ctx);
      if (is_plugin_disabled(jcr)) {
         Dmsg1(dbglvl, "Plugin %s disabled\n", cmd);
         goto bail_out;
      }
      if (!plug_func(plugin)->startRestoreFile || !plug_func(plugin)->endRestoreFile) {
         Dmsg1(dbglvl, "Plugin not suitable for Restore job %s\n", cmd);
         goto bail_out;
      }
      Dmsg1(dbglvl, "Restore Command plugin = %s\n", cmd);
      event.eventType = bEventRestoreCommand;
      if (plug_func(plugin)->handlePluginEvent(jcr->plugin_ctx,
                                               &event, cmd) != bRC_OK) {
         Dmsg1(dbglvl, "Handle event failed. Plugin=%s\n", cmd);
         goto bail_out;
      }
      if (bac_ctx->restoreFileStarted) {
         Jmsg2(jcr, M_FATAL, 0, "Second call to startRestoreFile. plugin=%s cmd=%s\n", plugin->file, cmd);
         bac_ctx->restoreFileStarted = false;
         goto bail_out;
      }
      if (plug_func(plugin)->startRestoreFile(jcr->plugin_ctx, cmd) == bRC_OK) {
         bac_ctx->restoreFileStarted = true;
         goto ok_out;
      } else {
         Dmsg1(dbglvl, "startRestoreFile failed. plugin=%s\n", cmd);
      }
      goto bail_out;
   }
   Jmsg1(jcr, M_WARNING, 0, _("Plugin=%s not found.\n"), cmd);
   goto bail_out;

ok_out:
   return start;

bail_out:
   Dsm_check(999);
   jcr->plugin = NULL;
   jcr->plugin_ctx = NULL;
   return start;
}

static void fill_restore_pkt(JCR *jcr, ATTR *attr, restore_pkt *rp)
{
   rp->pkt_size = sizeof(restore_pkt);
   rp->pkt_end = sizeof(restore_pkt);
   rp->delta_seq = attr->delta_seq;
   rp->stream = attr->stream;
   rp->data_stream = attr->data_stream;
   rp->type = attr->type;
   rp->file_index = attr->file_index;
   rp->LinkFI = attr->LinkFI;
   rp->uid = attr->uid;
   rp->statp = attr->statp;                /* structure assignment */
   rp->attrEx = attr->attrEx;
   if (jcr->is_JobType(JT_VERIFY)) {
      rp->ofname = attr->fname;
   } else {
      rp->ofname = attr->ofname;
      rp->olname = attr->olname;
   }
   rp->where = jcr->where;
   rp->RegexWhere = jcr->RegexWhere;
   rp->replace = jcr->replace;
   rp->create_status = CF_ERROR;
}

/**
 * Tell the plugin to create the file.  Return values are
 *   This is called only during Restore
 *
 *  CF_ERROR    -- error
 *  CF_SKIP     -- skip processing this file
 *  CF_EXTRACT  -- extract the file (i.e.call i/o routines)
 *  CF_CREATED  -- created, but no content to extract (typically directories)
 *
 */
int plugin_create_file(JCR *jcr, ATTR *attr, BFILE *bfd, int replace)
{
   bpContext *plugin_ctx = jcr->plugin_ctx;
   bacula_ctx *bac_ctx = get_bacula_ctx(plugin_ctx);
   Plugin *plugin = jcr->plugin;
   struct restore_pkt rp;
   int flags;
   int rc;

   Dsm_check(999);
   if (!plugin || !plugin_ctx || jcr->is_job_canceled()) {
      return CF_ERROR;
   }
   if (!plug_func(plugin)->createFile) {
      Dmsg0(dbglvl, "Plugin not suitable for Restore job\n");
      return CF_ERROR;
   }

   fill_restore_pkt(jcr, attr, &rp);

   Dmsg4(dbglvl, "call plugin createFile stream=%d type=%d LinkFI=%d File=%s\n",
         rp.stream, rp.type, rp.LinkFI, rp.ofname);
   if (rp.attrEx) {
      Dmsg1(dbglvl, "attrEx=\"%s\"\n", rp.attrEx);
   }
   Dsm_check(999);
   if (!bac_ctx->restoreFileStarted || bac_ctx->createFileCalled) {
      Jmsg2(jcr, M_FATAL, 0, "Unbalanced call to createFile=%d %d\n",
            bac_ctx->createFileCalled, bac_ctx->restoreFileStarted);
      bac_ctx->createFileCalled = false;
      return CF_ERROR;
   }
   rc = plug_func(plugin)->createFile(plugin_ctx, &rp);
   if (rc != bRC_OK) {
      Qmsg2(jcr, M_ERROR, 0, _("Plugin createFile call failed. Stat=%d file=%s\n"),
            rc, attr->ofname);
      return CF_ERROR;
   }
   if (rp.create_status == CF_ERROR) {
      Qmsg1(jcr, M_ERROR, 0, _("Plugin createFile call failed. Returned CF_ERROR file=%s\n"),
            attr->ofname);
      return CF_ERROR;
   }

   if (rp.create_status == CF_SKIP) {
      return CF_SKIP;
   }

   if (rp.create_status == CF_CORE) {
      return CF_CORE;           /* Let Bacula core handle the file creation */
   }

   /* Use the bfile for plugin */
   set_cmd_plugin(bfd, jcr);

   /* Created link or directory? */
   if (rp.create_status == CF_CREATED) {
      return rp.create_status;        /* yes, no need to bopen */
   }

   Dsm_check(999);

   flags =  O_WRONLY | O_CREAT | O_TRUNC | O_BINARY;
   Dmsg0(dbglvl, "call bopen\n");
   int stat = bopen(bfd, attr->ofname, flags, S_IRUSR | S_IWUSR);
   Dmsg1(dbglvl, "bopen status=%d\n", stat);
   if (stat < 0) {
      berrno be;
      be.set_errno(bfd->berrno);
      Qmsg2(jcr, M_ERROR, 0, _("Could not create %s: ERR=%s\n"),
            attr->ofname, be.bstrerror());
      Dmsg2(dbglvl,"Could not bopen file %s: ERR=%s\n", attr->ofname, be.bstrerror());
      return CF_ERROR;
   }

   if (!is_bopen(bfd)) {
      Dmsg0(000, "===== BFD is not open!!!!\n");
   }
   Dsm_check(999);
   return CF_EXTRACT;
}

/**
 * Reset the file attributes after all file I/O is done -- this allows
 *  the previous access time/dates to be set properly, and it also allows
 *  us to properly set directory permissions.
 *  Not currently Implemented.
 */
bool plugin_set_attributes(JCR *jcr, ATTR *attr, BFILE *ofd)
{
   Plugin *plugin = (Plugin *)jcr->plugin;
   struct restore_pkt rp;

   Dmsg0(dbglvl, "plugin_set_attributes\n");

   if (!plugin || !jcr->plugin_ctx) {
      return false;
   }

   memset(&rp, 0, sizeof(rp));
   rp.pkt_size = sizeof(rp);
   rp.pkt_end = sizeof(rp);
   rp.stream = attr->stream;
   rp.data_stream = attr->data_stream;
   rp.type = attr->type;
   rp.file_index = attr->file_index;
   rp.LinkFI = attr->LinkFI;
   rp.uid = attr->uid;
   rp.statp = attr->statp;                /* structure assignment */
   rp.attrEx = attr->attrEx;
   rp.ofname = attr->ofname;
   rp.olname = attr->olname;
   rp.where = jcr->where;
   rp.RegexWhere = jcr->RegexWhere;
   rp.replace = jcr->replace;
   rp.create_status = CF_ERROR;

   plug_func(plugin)->setFileAttributes(jcr->plugin_ctx, &rp);

   if (rp.create_status == CF_CORE) {
      set_attributes(jcr, attr, ofd);
   } else {
      if (is_bopen(ofd)) {
         bclose(ofd);
      }
      pm_strcpy(attr->ofname, "*none*");
   }

   Dsm_check(999);
   return true;
}

/*
 * The Plugin file's metadata backup.
 * Stream of metadata packets is being decoded, serialized and sent to the storage daemon for storing
 * on the volume. Each metadata packet type has its own STREAM_* type.
 *
 * in:
 *    jcr - Job Control Record
 *    ff_pkt - file save packet
 * out:
 *    true - backup completed successfully (or there was no metadata to backup at all)
 *    false - backup of metadata failed
 */
bool plugin_backup_metadata(JCR *jcr, FF_PKT *ff_pkt)
{
   /* Backup metadata if provided */
   if (ff_pkt->plug_meta) {
      if (!metadata_save(jcr, ff_pkt->plug_meta)) {
         Jmsg2(jcr, M_ERROR, 0, _("Failed to backup metadata for plugin: \"%s\" fname: %s\n"),
               jcr->plugin->file, ff_pkt->fname);
         return false;
      }
   }

   return true;
}

/*
 * The Plugin ACL data backup. We are using a new Plugin callback:
 *    handleXACLdata() for that. The new callback get a pointer to
 *    struct xacl_pkt as a main argument which consist of the following
 *    data:
 *       xacl.func - could be the one of BACL_BACKUP, BACL_RESTORE,
 *                   BXATTR_BACKUP, BXATTR_RESTORE
 *       xacl.count - the length of data at the content buffer
 *       xacl.content - the buffer itself
 *    The buffer (xacl.content) is supplied by Bacula during restore and has to
 *    be supplied by a Plugin during backup.
 *    The new callback should return bRC_OK on success and bRC_Error on
 *    any error.
 *
 * in:
 *    jcr - Job Control Record
 *    ff_pkt - file save packet
 *    data is a pointer to variable returned
 * out:
 *    data - the pointer to data buffer returned from plugin
 *    0 - Success, no more data to save
 *    > 0 - Success and the number of bytes returned in **data buffer
 *    -1 - Error, no acls data to backup
 */
int plugin_backup_acl(JCR *jcr, FF_PKT *ff_pkt, char **data)
{
   struct xacl_pkt xacl;
   Plugin *plugin = (Plugin *)jcr->plugin;
   bRC rc;

   Dmsg0(dbglvl, "plugin_backup_acl\n");

   /* check of input variables */
   if (!plugin || !jcr->plugin_ctx || !data) {
      return 0;
   }

   /* The plugin is not handling ACL/XATTR */
   if (plug_func(jcr->plugin)->handleXACLdata == NULL) {
      return 0;
   }
   
   /* prepare the xacl packet */
   memset(&xacl, 0, sizeof(xacl));
   xacl.pkt_size = sizeof(xacl);
   xacl.pkt_end = sizeof(xacl);
   xacl.func = BACL_BACKUP;

   rc = plug_func(plugin)->handleXACLdata(jcr->plugin_ctx, &xacl);

   /* check out status */
   if (rc != bRC_OK){
      Dmsg0(dbglvl, "plugin->handleXACLdata returned error\n");
      return -1;
   }
   if (xacl.count > 0){
      /* we have something to save, so prepare return data */
      *data = xacl.content;
      return xacl.count;
   }

   return 0;
}

/*
 * Called here when Bacula got ACL stream to restore but not every stream but
 *    a specific one: STREAM_XACL_PLUGIN_ACL which means a plugin has to
 *    be called.
 *
 * in:
 *    jcr - Job Control Record
 *    data - content to restore
 *    length - the length of the content to restore
 * out:
 *    true - when successful
 *    false - on any Error
 */
bool plugin_restore_acl(JCR *jcr, char *data, uint32_t length)
{
   struct xacl_pkt xacl;
   Plugin *plugin = (Plugin *)jcr->plugin;
   bRC rc;

   Dmsg0(dbglvl, "plugin_restore_acl\n");

   /* check of input variables */
   if (!plugin || !jcr->plugin_ctx || !data || length == 0) {
      return true;
   }

   /* The plugin is not handling ACL/XATTR */
   if (plug_func(jcr->plugin)->handleXACLdata == NULL) {
      return 0;
   }

   /* prepare the xacl packet */
   memset(&xacl, 0, sizeof(xacl));
   xacl.pkt_size = sizeof(xacl);
   xacl.pkt_end = sizeof(xacl);
   xacl.func = BACL_RESTORE;
   xacl.content = data;
   xacl.count = length;

   rc = plug_func(plugin)->handleXACLdata(jcr->plugin_ctx, &xacl);

   /* check out status */
   if (rc != bRC_OK){
      Dmsg0(dbglvl, "plugin->handleXACLdata returned error\n");
      return false;
   }

   return true;
}

/*
 * The Plugin XATTR data backup. We are using a new Plugin callback:
 *    handleXACLdata() for that. Check plugin_backup_acl for new callback
 *    description.
 *
 * in:
 *    jcr - Job Control Record
 *    ff_pkt - file save packet
 *    data is a pointer to variable returned
 * out:
 *    data - the pointer to data buffer returned from plugin
 *    0 - Success, no more data to save
 *    >0 - Success and the number of bytes returned in **data buffer
 *    <0 - Error
 */
int plugin_backup_xattr(JCR *jcr, FF_PKT *ff_pkt, char **data)
{

   struct xacl_pkt xacl;
   Plugin *plugin = (Plugin *)jcr->plugin;
   bRC rc;

   Dmsg0(dbglvl, "plugin_backup_xattr\n");

   /* check of input variables */
   if (!plugin || !jcr->plugin_ctx || !data) {
      return 0;
   }

   /* The plugin is not handling ACL/XATTR */
   if (plug_func(jcr->plugin)->handleXACLdata == NULL) {
      return 0;
   }

   /* prepare the xacl packet */
   memset(&xacl, 0, sizeof(xacl));
   xacl.pkt_size = sizeof(xacl);
   xacl.pkt_end = sizeof(xacl);
   xacl.func = BXATTR_BACKUP;

   rc = plug_func(plugin)->handleXACLdata(jcr->plugin_ctx, &xacl);

   /* check out status */
   if (rc != bRC_OK){
      Dmsg0(dbglvl, "plugin->handleXACLdata returned error\n");
      return -1;
   }
   if (xacl.count > 0){
      /* we have something to save, so prepare return data */
      *data = xacl.content;
      return xacl.count;
   }

   return 0;
}

/*
 * Called here when Bacula got XATTR stream to restore but not every stream but
 *    a specific one: STREAM_XACL_PLUGIN_XATTR which means a plugin has to
 *    be called.
 *
 * in:
 *    jcr - Job Control Record
 *    data - content to restore
 *    length - the length of the content to restore
 * out:
 *    true - when successful
 *    false - on any Error
 */
bool plugin_restore_xattr(JCR *jcr, char *data, uint32_t length)
{
   struct xacl_pkt xacl;
   Plugin *plugin = (Plugin *)jcr->plugin;
   bRC rc;

   Dmsg0(dbglvl, "plugin_restore_xattr\n");

   /* check of input variables */
   if (!plugin || !jcr->plugin_ctx || !data || length == 0) {
      return true;
   }

   /* The plugin is not handling ACL/XATTR */
   if (plug_func(jcr->plugin)->handleXACLdata == NULL) {
      return 0;
   }

   /* prepare the xacl packet */
   memset(&xacl, 0, sizeof(xacl));
   xacl.pkt_size = sizeof(xacl);
   xacl.pkt_end = sizeof(xacl);
   xacl.func = BXATTR_RESTORE;
   xacl.content = data;
   xacl.count = length;

   rc = plug_func(plugin)->handleXACLdata(jcr->plugin_ctx, &xacl);

   /* check out status */
   if (rc != bRC_OK){
      Dmsg0(dbglvl, "plugin->handleXACLdata returned error\n");
      return false;
   }

   return true;
}

/*
 * Allow plugin to modify stream 
 */
bool plugin_check_stream(JCR *jcr, int32_t &stream)
{
   struct stream_pkt sp;
   Plugin *plugin = (Plugin *)jcr->plugin;
   bRC rc;

   Dmsg0(dbglvl, "plugin_check_stream\n");

   /* check of input variables */
   if (!plugin || !jcr->plugin_ctx || !plug_func(plugin)->checkStream) {
      return true;
   }

   /* prepare the stream packet */
   memset(&sp, 0, sizeof(sp));
   sp.pkt_size = sizeof(sp);
   sp.stream = stream;
   sp.pkt_end = sizeof(sp);

   rc = plug_func(plugin)->checkStream(jcr->plugin_ctx, &sp);

   /* check out status */
   if (rc != bRC_OK){
      Dmsg0(dbglvl, "plugin->checkStream returned error\n");
      return false;
   }
   stream = sp.stream;

   return true;
}

/*
 * Query the plugin about a parameter
 */
bool plugin_query_parameter(JCR *jcr, char *param, char *command, void sendit(JCR *jcr, const char *str))
{
   query_pkt query;
   Plugin *plugin;
   bRC rc;
   int len;
   int i;
   int found=0;

   Dmsg0(dbglvl, "plugin_query\n");

   if (!b_plugin_list || !get_plugin_name(jcr, command, &len)) {
      goto bail_out;
   }

   /* Note, we stop the loop on the first plugin that matches the name */
   foreach_alist_index(i, plugin, b_plugin_list) {
      if (!for_this_plugin(plugin, command, len)) {
         continue;
      }
      jcr->plugin_ctx = &jcr->plugin_ctx_list[i];
      jcr->plugin = (Plugin*)b_plugin_list->get(i);

      if (is_plugin_disabled(jcr->plugin_ctx)) {
         Dmsg1(50, "Plugin %s disabled\n", plugin->file);
         continue;
      }

      /* check of input variables */
      if (!jcr->plugin || !jcr->plugin_ctx || !plug_func(plugin)->queryParameter) {
         break;
      }

      memset(&query, 0, sizeof(query));
      query.pkt_size = sizeof(query);
      query.command = command;
      query.parameter = param;
      query.pkt_end = sizeof(query);
   
      /* The output is returned via Jmsg */
      do {
         /* memory allocated inside the plugin context */
         rc = plug_func(plugin)->queryParameter(jcr->plugin_ctx, &query);

         if (rc == bRC_More || rc == bRC_OK) {
            sendit(jcr, query.result);
         }         
      } while (rc == bRC_More);

      /* check out status */
      if (rc != bRC_OK){
         Dmsg0(dbglvl, "plugin->queryParameter returned error\n");
         OutputWriter ow;// TODO: pass api_opts
         sendit(jcr, ow.get_output(OT_STRING, "error", "plugin protocol error", OT_END));
         goto bail_out;
      }
      found=1;
      break;
   }
   if (!found) {
      OutputWriter ow;        // TODO: pass api_opts
      sendit(jcr, ow.get_output(OT_STRING, "error", "plugin not found", OT_END));
   }
bail_out:
   jcr->plugin_ctx = NULL;
   jcr->plugin = NULL;
   return true;
}

/*
 * Print to file the plugin info.
 */
void dump_fd_plugin(Plugin *plugin, FILE *fp)
{
   if (!plugin) {
      return ;
   }
   pInfo *info = (pInfo *)plugin->pinfo;
   fprintf(fp, "\tversion=%d\n", info->version);
   fprintf(fp, "\tdate=%s\n", NPRTB(info->plugin_date));
   fprintf(fp, "\tmagic=%s\n", NPRTB(info->plugin_magic));
   fprintf(fp, "\tauthor=%s\n", NPRTB(info->plugin_author));
   fprintf(fp, "\tlicence=%s\n", NPRTB(info->plugin_license));
   fprintf(fp, "\tversion=%s\n", NPRTB(info->plugin_version));
   fprintf(fp, "\tdescription=%s\n", NPRTB(info->plugin_description));
}

/**
 * This entry point is called internally by Bacula to ensure
 *  that the plugin IO calls come into this code.
 */
void load_fd_plugins(const char *plugin_dir)
{
   Plugin *plugin;
   int i;

   if (!plugin_dir) {
      Dmsg0(dbglvl, "plugin dir is NULL\n");
      return;
   }

   b_plugin_list = New(alist(10, not_owned_by_alist));
   Dsm_check(999);
   if (!load_plugins((void *)&binfo, (void *)&bfuncs, plugin_dir, plugin_type,
                     is_plugin_compatible)) {
      /* Either none found, or some error */
      if (b_plugin_list->size() == 0) {
         delete b_plugin_list;
         b_plugin_list = NULL;
         Dmsg0(dbglvl, "No plugins loaded\n");
         return;
      }
   }

   /* Plug entry points called from findlib */
   plugin_bopen  = my_plugin_bopen;
   plugin_bclose = my_plugin_bclose;
   plugin_bread  = my_plugin_bread;
   plugin_bwrite = my_plugin_bwrite;
   plugin_blseek = my_plugin_blseek;
   Dsm_check(999);

   /*
    * Verify that the plugin is acceptable, and print information
    *  about it.
    */
   foreach_alist_index(i, plugin, b_plugin_list) {
      Jmsg(NULL, M_INFO, 0, _("Loaded plugin: %s\n"), plugin->file);
      Dmsg1(dbglvl, "Loaded plugin: %s\n", plugin->file);
   }

   dbg_plugin_add_hook(dump_fd_plugin);
   Dsm_check(999);
}

/**
 * Check if a plugin is compatible.  Called by the load_plugin function
 *  to allow us to verify the plugin.
 */
static bool is_plugin_compatible(Plugin *plugin)
{
   pInfo *info = (pInfo *)plugin->pinfo;
   Dmsg0(dbglvl, "is_plugin_compatible called\n");
   Dsm_check(999);
   if (chk_dbglvl(50)) {
      dump_fd_plugin(plugin, stdin);
   }
   if (strcmp(info->plugin_magic, FD_PLUGIN_MAGIC) != 0) {
      Jmsg(NULL, M_ERROR, 0, _("Plugin magic wrong. Plugin=%s wanted=%s got=%s\n"),
           plugin->file, FD_PLUGIN_MAGIC, info->plugin_magic);
      Dmsg3(50, "Plugin magic wrong. Plugin=%s wanted=%s got=%s\n",
           plugin->file, FD_PLUGIN_MAGIC, info->plugin_magic);

      return false;
   }
   if (info->version != FD_PLUGIN_INTERFACE_VERSION) {
      Jmsg(NULL, M_ERROR, 0, _("Plugin version incorrect. Plugin=%s wanted=%d got=%d\n"),
           plugin->file, FD_PLUGIN_INTERFACE_VERSION, info->version);
      Dmsg3(50, "Plugin version incorrect. Plugin=%s wanted=%d got=%d\n",
           plugin->file, FD_PLUGIN_INTERFACE_VERSION, info->version);
      return false;
   }
   if (strcmp(info->plugin_license, "Bacula AGPLv3") != 0 &&
       strcmp(info->plugin_license, "AGPLv3") != 0 &&
       strcmp(info->plugin_license, BPLUGIN_LICENSE) != 0) {
      Jmsg(NULL, M_ERROR, 0, _("Plugin license incompatible. Plugin=%s license=%s\n"),
           plugin->file, info->plugin_license);
      Dmsg2(50, "Plugin license incompatible. Plugin=%s license=%s\n",
           plugin->file, info->plugin_license);
      return false;
   }
   if (info->size != sizeof(pInfo)) {
      Jmsg(NULL, M_ERROR, 0,
           _("Plugin size incorrect. Plugin=%s wanted=%d got=%d\n"),
           plugin->file, sizeof(pInfo), info->size);
      return false;
   }

   Dsm_check(999);
   return true;
}


/**
 * Create a new instance of each plugin for this Job
 *   Note, b_plugin_list can exist but jcr->plugin_ctx_list can
 *   be NULL if no plugins were loaded.
 */
void new_plugins(JCR *jcr)
{
   Plugin *plugin;
   int i;

   Dsm_check(999);
   if (!b_plugin_list) {
      Dmsg0(dbglvl, "plugin list is NULL\n");
      return;
   }
   if (jcr->is_job_canceled()) {
      return;
   }

   int num = b_plugin_list->size();

   if (num == 0) {
      Dmsg0(dbglvl, "No plugins loaded\n");
      return;
   }

   jcr->plugin_ctx_list = (bpContext *)malloc(sizeof(bpContext) * num);

   bpContext *plugin_ctx_list = (bpContext *)jcr->plugin_ctx_list;
   Dmsg2(dbglvl, "Instantiate plugin_ctx=%p JobId=%d\n", plugin_ctx_list, jcr->JobId);
   foreach_alist_index(i, plugin, b_plugin_list) {
      Dsm_check(999);
      /* Start a new instance of each plugin */
      bacula_ctx *b_ctx = (bacula_ctx *)malloc(sizeof(bacula_ctx));
      memset(b_ctx, 0, sizeof(bacula_ctx));
      b_ctx->jcr = jcr;
      b_ctx->plugin = plugin;
      plugin_ctx_list[i].bContext = (void *)b_ctx;   /* Bacula private context */
      plugin_ctx_list[i].pContext = NULL;
      if (plug_func(plugin)->newPlugin(&plugin_ctx_list[i]) != bRC_OK) {
         Dmsg1(000, "Plugin %s will be disabled\n", plugin->file);
         b_ctx->disabled = true;
      }
   }
   if (i > num) {
      Jmsg2(jcr, M_ABORT, 0, "Num plugins=%d exceeds list size=%d\n",
            i, num);
   }
   Dsm_check(999);
}

/**
 * Free the plugin instances for this Job
 */
void free_plugins(JCR *jcr)
{
   Plugin *plugin;
   int i;

   if (!b_plugin_list || !jcr->plugin_ctx_list) {
      return;                         /* no plugins, nothing to do */
   }

   Dsm_check(999);
   bpContext *plugin_ctx_list = (bpContext *)jcr->plugin_ctx_list;
   Dmsg2(dbglvl, "Free instance plugin_ctx=%p JobId=%d\n", plugin_ctx_list, jcr->JobId);
   foreach_alist_index(i, plugin, b_plugin_list) {
      /* Free the plugin instance */
      plug_func(plugin)->freePlugin(&plugin_ctx_list[i]);
      free(plugin_ctx_list[i].bContext);     /* free Bacula private context */
      Dsm_check(999);
   }
   Dsm_check(999);
   free(plugin_ctx_list);
   jcr->plugin_ctx_list = NULL;
}

static int my_plugin_bopen(BFILE *bfd, const char *fname, uint64_t flags, mode_t mode)
{
   JCR *jcr = bfd->jcr;
   Plugin *plugin = (Plugin *)jcr->plugin;
   struct io_pkt io;

   Dmsg1(dbglvl, "plugin_bopen flags=%x\n", flags);
   Dsm_check(999);
   if (!plugin || !jcr->plugin_ctx) {
      return 0;
   }
   io.pkt_size = sizeof(io);
   io.pkt_end = sizeof(io);
   io.func = IO_OPEN;
   io.count = 0;
   io.buf = NULL;
   io.fname = fname;
   io.flags = flags;
   io.mode = mode;
   io.win32 = false;
   io.lerror = 0;
   io.status = -1;
   plug_func(plugin)->pluginIO(jcr->plugin_ctx, &io);
   bfd->berrno = io.io_errno;
   if (io.win32) {
      errno = b_errno_win32;
   } else {
      errno = io.io_errno;
      bfd->lerror = io.lerror;
   }
   Dmsg1(dbglvl, "Return from plugin open status=%d\n", io.status);
   Dsm_check(999);
   return io.status;
}

static int my_plugin_bclose(BFILE *bfd)
{
   JCR *jcr = bfd->jcr;
   Plugin *plugin = (Plugin *)jcr->plugin;
   struct io_pkt io;

   Dsm_check(999);
   Dmsg0(dbglvl, "===== plugin_bclose\n");
   if (!plugin || !jcr->plugin_ctx) {
      return 0;
   }
   /* We can be canceled after a network error, we need to notify the plugin */
   if (jcr->is_canceled()) {
      generate_plugin_event(jcr, bEventCancelCommand);
   }
   io.pkt_size = sizeof(io);
   io.pkt_end = sizeof(io);
   io.func = IO_CLOSE;
   io.count = 0;
   io.buf = NULL;
   io.win32 = false;
   io.lerror = 0;
   io.status = -1;
   plug_func(plugin)->pluginIO(jcr->plugin_ctx, &io);
   bfd->berrno = io.io_errno;
   if (io.win32) {
      errno = b_errno_win32;
   } else {
      errno = io.io_errno;
      bfd->lerror = io.lerror;
   }
   Dmsg1(dbglvl, "plugin_bclose stat=%d\n", io.status);
   Dsm_check(999);
   return io.status;
}

static ssize_t my_plugin_bread(BFILE *bfd, void *buf, size_t count)
{
   JCR *jcr = bfd->jcr;
   Plugin *plugin = (Plugin *)jcr->plugin;
   struct io_pkt io;

   Dsm_check(999);
   Dmsg0(dbglvl2, "plugin_bread\n");
   if (!plugin || !jcr->plugin_ctx) {
      return 0;
   }
   io.pkt_size = sizeof(io);
   io.pkt_end = sizeof(io);
   io.func = IO_READ;
   io.count = count;
   io.buf = (char *)buf;
   io.win32 = false;
   io.offset = 0;
   io.lerror = 0;
   io.status = -1;
   plug_func(plugin)->pluginIO(jcr->plugin_ctx, &io);
   bfd->offset = io.offset;
   bfd->berrno = io.io_errno;
   if (io.win32) {
      errno = b_errno_win32;
   } else {
      errno = io.io_errno;
      bfd->lerror = io.lerror;
   }
   Dsm_check(999);
   return (ssize_t)io.status;
}

static ssize_t my_plugin_bwrite(BFILE *bfd, void *buf, size_t count)
{
   JCR *jcr = bfd->jcr;
   Plugin *plugin = (Plugin *)jcr->plugin;
   struct io_pkt io;

   Dsm_check(999);
   Dmsg0(dbglvl2, "plugin_bwrite\n");
   if (!plugin || !jcr->plugin_ctx) {
      Dmsg0(dbglvl, "No plugin context\n");
      return 0;
   }
   io.pkt_size = sizeof(io);
   io.pkt_end = sizeof(io);
   io.func = IO_WRITE;
   io.count = count;
   io.buf = (char *)buf;
   io.win32 = false;
   io.lerror = 0;
   io.status = -1;
   plug_func(plugin)->pluginIO(jcr->plugin_ctx, &io);
   bfd->berrno = io.io_errno;
   if (io.win32) {
      errno = b_errno_win32;
   } else {
      errno = io.io_errno;
      bfd->lerror = io.lerror;
   }
   Dsm_check(999);
   return (ssize_t)io.status;
}

static boffset_t my_plugin_blseek(BFILE *bfd, boffset_t offset, int whence)
{
   JCR *jcr = bfd->jcr;
   Plugin *plugin = (Plugin *)jcr->plugin;
   struct io_pkt io;

   Dsm_check(999);
   Dmsg0(dbglvl, "plugin_bseek\n");
   if (!plugin || !jcr->plugin_ctx) {
      return 0;
   }
   io.pkt_size = sizeof(io);
   io.pkt_end = sizeof(io);
   io.func = IO_SEEK;
   io.offset = offset;
   io.whence = whence;
   io.win32 = false;
   io.lerror = 0;
   plug_func(plugin)->pluginIO(jcr->plugin_ctx, &io);
   bfd->berrno = io.io_errno;
   if (io.win32) {
      errno = b_errno_win32;
   } else {
      errno = io.io_errno;
      bfd->lerror = io.lerror;
   }
   Dsm_check(999);
   return (boffset_t)io.offset;
}

/* ==============================================================
 *
 * Callbacks from the plugin
 *
 * ==============================================================
 */
static bRC baculaGetValue(bpContext *ctx, bVariable var, void *value)
{
   JCR *jcr;
   if (!value) {
      return bRC_Error;
   }

   Dsm_check(999);
   switch (var) {               /* General variables, no need of ctx */
   case bVarFDName:
      *((char **)value) = my_name;
      break;
   case bVarWorkingDir:
      *(void **)value = me->working_directory;
      break;
   case bVarPluginDir:
      *(void **)value = me->plugin_directory;
      break;
   case bVarExePath:
      *(char **)value = exepath;
      break;
   case bVarSysConfigPath:
      *(char **)value = sysconfigpath;
      break;
   case bVarVersion:
      *(char **)value = version;
      break;
   case bVarDistName:
      *(char **)value = dist_name;
      break;
   case bVarPrevJobName:
      break;
   case bVarPrefixLinks:
      break;
   default:
      break;
   }

   if (!ctx) {                  /* Other variables need context */
      return bRC_Error;
   }

   jcr = ((bacula_ctx *)ctx->bContext)->jcr;
   if (!jcr) {
      return bRC_Error;
   }

   switch (var) {
   case bVarIsCanceled:
      *((int *)value) = jcr->is_canceled();
      Dmsg1(dbglvl, "Bacula: return bVarIsCanceled=%d\n", jcr->is_canceled());
      break;
   case bVarJobId:
      *((int *)value) = jcr->JobId;
      Dmsg1(dbglvl, "Bacula: return bVarJobId=%d\n", jcr->JobId);
      break;
   case bVarLevel:
      *((int *)value) = jcr->getJobLevel();
      Dmsg1(dbglvl, "Bacula: return bVarJobLevel=%d\n", jcr->getJobLevel());
      break;
   case bVarType:
      *((int *)value) = jcr->getJobType();
      Dmsg1(dbglvl, "Bacula: return bVarJobType=%d\n", jcr->getJobType());
      break;
   case bVarClient:
      *((char **)value) = jcr->client_name;
      Dmsg1(dbglvl, "Bacula: return Client_name=%s\n", jcr->client_name);
      break;
   case bVarJobName:
      *((char **)value) = jcr->Job;
      Dmsg1(dbglvl, "Bacula: return Job name=%s\n", jcr->Job);
      break;
   case bVarPrevJobName:
      *((char **)value) = jcr->PrevJob;
      Dmsg1(dbglvl, "Bacula: return Previous Job name=%s\n", jcr->PrevJob);
      break;
   case bVarJobStatus:
      *((int *)value) = jcr->JobStatus;
      Dmsg1(dbglvl, "Bacula: return bVarJobStatus=%d\n", jcr->JobStatus);
      break;
   case bVarSinceTime:
      *((int *)value) = (int)jcr->mtime;
      Dmsg1(dbglvl, "Bacula: return since=%d\n", (int)jcr->mtime);
      break;
   case bVarAccurate:
      *((int *)value) = (int)jcr->accurate;
      Dmsg1(dbglvl, "Bacula: return accurate=%d\n", (int)jcr->accurate);
      break;
   case bVarInteractiveSession:
      *(int *)value = (int)jcr->interactive_session;
      break;
   case bVarFileIndex:
      *(int *)value = (int)jcr->JobFiles;
      break;
   case bVarFileSeen:
      break;                 /* a write only variable, ignore read request */
   case bVarVssObject:
#ifdef HAVE_WIN32
      if (jcr->pVSSClient) {
         *(void **)value = jcr->pVSSClient->GetVssObject();
         break;
       }
#endif
      *(void **)value = NULL;
       return bRC_Error;
   case bVarVssDllHandle:
#ifdef HAVE_WIN32
      *(void **)value = vsslib;
      break;
#endif
       return bRC_Error;
   case bVarWhere:
      *(char **)value = jcr->where;
      break;
   case bVarRegexWhere:
      *(char **)value = jcr->RegexWhere;
      break;
   case bVarPrefixLinks:
      *(int *)value = (int)jcr->prefix_links;
      break;
   case bVarReplace:
      *((int*)value) = jcr->replace;
      Dmsg1(dbglvl, "Bacula: return replace=%c\n", jcr->replace);
      break;
   case bVarMaxDedupBlockSize:
      *((int*)value) = jcr->max_dedup_block_size;
      break;
   case bVarMinDedupBlockSize:
      *((int*)value) = jcr->min_dedup_block_size;
      break;
   case bVarEstimate:
      *((int*)value) = jcr->estimate?1:0;
      break;
   case bVarFDName:             /* get warning with g++ if we missed one */
   case bVarWorkingDir:
   case bVarPluginDir:
   case bVarExePath:
   case bVarSysConfigPath:
   case bVarVersion:
   case bVarDistName:
      break;
   }
   Dsm_check(999);
   return bRC_OK;
}

static bRC baculaSetValue(bpContext *ctx, bVariable var, void *value)
{
   JCR *jcr;
   Dsm_check(999);
   if (!value || !ctx) {
      return bRC_Error;
   }
// Dmsg1(dbglvl, "bacula: baculaGetValue var=%d\n", var);
   jcr = ((bacula_ctx *)ctx->bContext)->jcr;
   if (!jcr) {
      return bRC_Error;
   }
// Dmsg1(dbglvl, "Bacula: jcr=%p\n", jcr);
   switch (var) {
   case bVarFileSeen:
      if (!accurate_mark_file_as_seen(jcr, (char *)value)) {
         return bRC_Error;
      }
      break;
   case bVarInteractiveSession:
      jcr->interactive_session = (((intptr_t) value) == 1);
      break;
   default:
      break;
   }
   Dsm_check(999);
   return bRC_OK;
}

static bRC baculaRegisterEvents(bpContext *ctx, ...)
{
   va_list args;
   uint32_t event;

   va_start(args, ctx);
   while ((event = va_arg(args, uint32_t))) {
      Dmsg1(dbglvl, "Plugin wants event=%u\n", event);
      switch (event) {
      case bEventVerifyStream:
         /* We will call specific function, maybe VerifyFile() + pluginIO */
         plugin_register_verify_data(ctx);
         break;
      default:
         Dmsg0(50, "Event registration not implemented\n");
         break;
      }
   }
   va_end(args);
   Dsm_check(999);
   return bRC_OK;
}

static bRC baculaJobMsg(bpContext *ctx, const char *file, int line,
  int type, utime_t mtime, const char *fmt, ...)
{
   va_list arg_ptr;
   char buf[2000];
   JCR *jcr;

   Dsm_check(999);
   if (ctx) {
      jcr = ((bacula_ctx *)ctx->bContext)->jcr;
   } else {
      jcr = NULL;
   }

   va_start(arg_ptr, fmt);
   bvsnprintf(buf, sizeof(buf), fmt, arg_ptr);
   va_end(arg_ptr);
   Jmsg(jcr, type, mtime, "%s", buf);
   Dsm_check(999);
   return bRC_OK;
}

static bRC baculaDebugMsg(bpContext *ctx, const char *file, int line,
  int level, const char *fmt, ...)
{
   va_list arg_ptr;
   char buf[2000];

   Dsm_check(999);
   va_start(arg_ptr, fmt);
   bvsnprintf(buf, sizeof(buf), fmt, arg_ptr);
   va_end(arg_ptr);
   d_msg(file, line, level, "%s", buf);
   Dsm_check(999);
   return bRC_OK;
}

static void *baculaMalloc(bpContext *ctx, const char *file, int line,
              size_t size)
{
#ifdef SMARTALLOC
   return sm_malloc(file, line, size);
#else
   return malloc(size);
#endif
}

static void baculaFree(bpContext *ctx, const char *file, int line, void *mem)
{
#ifdef SMARTALLOC
   sm_free(file, line, mem);
#else
   free(mem);
#endif
}

static bool is_ctx_good(bpContext *ctx, JCR *&jcr, bacula_ctx *&bctx)
{
   Dsm_check(999);
   if (!ctx) {
      return false;
   }
   bctx = (bacula_ctx *)ctx->bContext;
   if (!bctx) {
      return false;
   }
   jcr = bctx->jcr;
   if (!jcr) {
      return false;
   }
   return true;
}

/**
 * Let the plugin define files/directories to be excluded
 *  from the main backup.
 */
static bRC baculaAddExclude(bpContext *ctx, const char *file)
{
   JCR *jcr;
   findINCEXE *old;
   bacula_ctx *bctx;
   Dsm_check(999);
   if (!is_ctx_good(ctx, jcr, bctx)) {
      return bRC_Error;
   }
   if (!file) {
      return bRC_Error;
   }

   /* Save the include context */
   old = get_incexe(jcr);

   /* Not right time to add exlude */
   if (!old) {
      return bRC_Error;
   }

   if (!bctx->exclude) {
      bctx->exclude = new_exclude(jcr);
   }

   /* Set the Exclude context */
   set_incexe(jcr, bctx->exclude);

   if (add_file_to_fileset(jcr, file, true) != state_include) {
      Dmsg1(100, "Failed to add exclude file=%s\n", file);
      return bRC_Error;
   }

   /* Restore the current context */
   set_incexe(jcr, old);

   Dmsg1(100, "Add exclude file=%s\n", file);
   Dsm_check(999);
   return bRC_OK;
}

/**
 * Let the plugin define files/directories to be included
 *  from the main backup.
 */
static bRC baculaAddInclude(bpContext *ctx, const char *file)
{
   JCR *jcr;
   findINCEXE *old;
   bacula_ctx *bctx;

   Dsm_check(999);
   if (!is_ctx_good(ctx, jcr, bctx)) {
      return bRC_Error;
   }
   if (!file) {
      return bRC_Error;
   }

   /* Save the include context */
   old = get_incexe(jcr);

   /* Not right time to add include */
   if (!old) {
      return bRC_Error;
   }
   if (!bctx->include) {
      bctx->include = old;
   }

   set_incexe(jcr, bctx->include);
   if (add_file_to_fileset(jcr, file, true) != state_include) {
      Dmsg1(100, "Failed to add include file=%s\n", file);
      return bRC_Error;
   }

   /* Restore the current context */
   set_incexe(jcr, old);

   Dmsg1(100, "Add include file=%s\n", file);
   Dsm_check(999);
   return bRC_OK;
}

/**
 * Let the plugin create FileEvent to report problems on files
 */
static bRC baculaAddFileEvent(bpContext *ctx, struct fileevent_pkt *event)
{
   JCR *jcr;
   bacula_ctx *bctx;
   struct fileevent_pkt *e;
   Dsm_check(999);
   if (!is_ctx_good(ctx, jcr, bctx)) {
      return bRC_Error;
   }
   // Sanity check
   if (!event) {
      return bRC_Error;
   }
   e = (struct fileevent_pkt*) malloc(sizeof(struct fileevent_pkt));
   memcpy(e, event, sizeof(struct fileevent_pkt));
   jcr->fileevents->append(e);
   
   Dsm_check(999);
   return bRC_OK;
}

/**
 * Let the plugin define plugin to be included
 *  from the main backup.
 */
static bRC baculaAddPlugin(bpContext *ctx, const char *file)
{
   JCR *jcr;
   findINCEXE *old;
   bacula_ctx *bctx;

   Dsm_check(999);
   if (!is_ctx_good(ctx, jcr, bctx)) {
      return bRC_Error;
   }
   if (!file) {
      return bRC_Error;
   }

   /* Save the include context */
   old = get_incexe(jcr);

   /* Not right time to add include */
   if (!old) {
      return bRC_Error;
   }
   if (!bctx->include) {
      bctx->include = old;
   }

   set_incexe(jcr, bctx->include);
   if (add_file_to_fileset(jcr, file, false) != state_include) {
      Dmsg1(100, "Failed to plugin=%s\n", file);
      return bRC_Error;
   }

   /* Restore the current context */
   set_incexe(jcr, old);

   Dmsg1(100, "Add include file=%s\n", file);
   Dsm_check(999);
   return bRC_OK;
}

static bRC baculaAddOptions(bpContext *ctx, const char *opts)
{
   JCR *jcr;
   bacula_ctx *bctx;
   Dsm_check(999);
   if (!is_ctx_good(ctx, jcr, bctx)) {
      return bRC_Error;
   }
   if (!opts) {
      return bRC_Error;
   }
   add_options_to_fileset(jcr, opts);
   Dsm_check(999);
   Dmsg1(1000, "Add options=%s\n", opts);
   return bRC_OK;
}

static bRC baculaAddRegex(bpContext *ctx, const char *item, int type)
{
   JCR *jcr;
   bacula_ctx *bctx;
   Dsm_check(999);
   if (!is_ctx_good(ctx, jcr, bctx)) {
      return bRC_Error;
   }
   if (!item) {
      return bRC_Error;
   }
   add_regex_to_fileset(jcr, item, type);
   Dmsg1(100, "Add regex=%s\n", item);
   Dsm_check(999);
   return bRC_OK;
}

static bRC baculaAddWild(bpContext *ctx, const char *item, int type)
{
   JCR *jcr;
   bacula_ctx *bctx;
   Dsm_check(999);
   if (!is_ctx_good(ctx, jcr, bctx)) {
      return bRC_Error;
   }
   if (!item) {
      return bRC_Error;
   }
   add_wild_to_fileset(jcr, item, type);
   Dmsg1(100, "Add wild=%s\n", item);
   Dsm_check(999);
   return bRC_OK;
}

static bRC baculaNewOptions(bpContext *ctx)
{
   JCR *jcr;
   bacula_ctx *bctx;
   Dsm_check(999);
   if (!is_ctx_good(ctx, jcr, bctx)) {
      return bRC_Error;
   }
   (void)new_options(jcr, NULL);
   Dsm_check(999);
   return bRC_OK;
}

static bRC baculaNewInclude(bpContext *ctx)
{
   JCR *jcr;
   bacula_ctx *bctx;
   Dsm_check(999);
   if (!is_ctx_good(ctx, jcr, bctx)) {
      return bRC_Error;
   }
   // without this any AddInclude() is unable to use more than one Include{} block
   bctx->include = new_include(jcr);
   Dsm_check(999);
   return bRC_OK;
}

static bRC baculaNewPreInclude(bpContext *ctx)
{
   JCR *jcr;
   bacula_ctx *bctx;
   Dsm_check(999);
   if (!is_ctx_good(ctx, jcr, bctx)) {
      return bRC_Error;
   }

   bctx->include = new_preinclude(jcr);
   new_options(jcr, bctx->include);
   set_incexe(jcr, bctx->include);

   Dsm_check(999);
   return bRC_OK;
}

/*
 * Check if a file have to be backuped using Accurate code
 */
static bRC baculaCheckChanges(bpContext *ctx, struct save_pkt *sp)
{
   JCR *jcr;
   bacula_ctx *bctx;
   FF_PKT *ff_pkt;
   bRC ret = bRC_Error;

   Dsm_check(999);
   if (!is_ctx_good(ctx, jcr, bctx)) {
      goto bail_out;
   }
   if (!sp) {
      goto bail_out;
   }

   ff_pkt = jcr->ff;
   /*
    * Copy fname and link because save_file() zaps them.  This
    *  avoids zaping the plugin's strings.
    */
   ff_pkt->type = sp->type;
   if (!sp->fname) {
      Jmsg0(jcr, M_FATAL, 0, _("Command plugin: no fname in baculaCheckChanges packet.\n"));
      goto bail_out;
   }

   ff_pkt->fname = sp->fname;
   ff_pkt->link = sp->link;
   memcpy(&ff_pkt->statp, &sp->statp, sizeof(ff_pkt->statp));

   if (check_changes(jcr, ff_pkt))  {
      ret = bRC_OK;
   } else {
      ret = bRC_Seen;
   }

   /* check_changes() can update delta sequence number, return it to the
    * plugin
    */
   sp->delta_seq = ff_pkt->delta_seq;
   sp->accurate_found = ff_pkt->accurate_found;

bail_out:
   Dsm_check(999);
   Dmsg1(100, "checkChanges=%i\n", ret);
   return ret;
}

/*
 * Check if a file would be saved using current Include/Exclude code
 */
static bRC baculaAcceptFile(bpContext *ctx, struct save_pkt *sp)
{
   JCR *jcr;
   FF_PKT *ff_pkt;
   bacula_ctx *bctx;

   char *old;
   struct stat oldstat;
   bRC ret = bRC_Error;

   Dsm_check(999);
   if (!is_ctx_good(ctx, jcr, bctx)) {
      goto bail_out;
   }
   if (!sp) {
      goto bail_out;
   }

   ff_pkt = jcr->ff;

   /* Probably not needed, but keep a copy */
   old = ff_pkt->fname;
   oldstat = ff_pkt->statp;

   ff_pkt->fname = sp->fname;
   ff_pkt->statp = sp->statp;

   if (accept_file(jcr, ff_pkt)) {
      ret = bRC_OK;
   } else {
      ret = bRC_Skip;
   }

   ff_pkt->fname = old;
   ff_pkt->statp = oldstat;

bail_out:
   return ret;
}

/**
 * @brief Get features from plugins, we fill a alist with {plugin name, features list}
 *
 * @param JCR *jcr
 * @param alist *ret: list of the features available
 * @return bool
 */
bool plugin_get_features(JCR *jcr, alist *ret)
{
   bFeature *elt;
   bpContext *plugin_ctx, *plugin_ctx_list;
   bEvent event;
   Plugin *plugin;
   const char *features;
   int i;

   event.eventType = bEventFeatures;
   plugin_ctx_list = (bpContext *)jcr->plugin_ctx_list;

   if (!plugin_ctx_list) {      /* Nothing to report */
      return true;
   }
   
   /* Call each plugin with the bEventFeatures */
   foreach_alist_index(i, plugin, b_plugin_list) {
      plugin_ctx = &plugin_ctx_list[i];
      if (is_plugin_disabled(plugin_ctx)) {
         Dmsg1(50, "Plugin %s disabled\n", plugin->file);
         continue;
      }
      features = NULL;
      plug_func(plugin)->handlePluginEvent(plugin_ctx, &event, &features);
      if (features) {           /* TODO: Check the features string */
         elt = (bFeature*) malloc(sizeof(bFeature));
         elt->plugin = plugin->file;
         elt->features = features;
         ret->append(elt);
      }
   }
   return true;
}

/**
 * @brief Bacula Callback to get accurate data from the table.
 *
 * @param ctx Bacula plugin context
 * @param att the in/out query parameter and data response; the input is att->fname
 * @return bRC when bRC_Seen then query was success and found some data,
 *             bRC_OK when no data found or no accurate data is available
 *             bRC_Error on any error found
 */
static bRC baculaAccurateAttribs(bpContext *ctx, accurate_attribs_pkt *att)
{
   JCR *jcr;
   bacula_ctx *bctx;

   Dsm_check(999);
   if (!is_ctx_good(ctx, jcr, bctx) || att == NULL) {
      return bRC_Error;
   }

   if (accurate_get_file_attribs(jcr, att)) {
      return bRC_Seen;
   }

   return bRC_OK;
}

/**
 * @brief Verify Data Plugin function to close a stream. The plugin is responsible to display a message.
 *
 * @param JCR *jcr
 * @return bRC bRC_Error when a plugin has returned an error during the pluginIO(close) or the endVerifyFile()
 *             bRC_OK when all plugins are OK
 */
bRC plugin_verify_data_close(JCR *jcr)
{
   bpContext *ctx;
   bacula_ctx *bctx;
   struct io_pkt io;
   bRC ret = bRC_OK;
   if (!jcr->plugin_verify) {
      return bRC_Skip;              // Nothing registered
   }
   foreach_alist(ctx, jcr->plugin_verify) {
      bctx = get_bacula_ctx(ctx);
      jcr->plugin_ctx = ctx;
      jcr->plugin = bctx->plugin;
      if (is_plugin_disabled(jcr)) {
         continue;
      }
      if (bctx->verifyFileCalled) {
         io.pkt_size = sizeof(io);
         io.pkt_end = sizeof(io);
         io.func = IO_CLOSE;
         io.count = 0;
         io.buf = NULL;
         io.fname = NULL;
         io.flags = 0;
         io.mode = O_WRONLY;
         io.win32 = false;
         io.lerror = 0;
         io.status = -1;
         plug_func(jcr->plugin)->pluginIO(jcr->plugin_ctx, &io);
         if (io.win32) {
            errno = b_errno_win32;
         } else {
            errno = io.io_errno;
         }
         bctx->verifyFileCalled = false;
         if (io.status != 0) {
	    Dmsg2(50, "Plugin %s in pluginIO(CLOSE) status=%d\n", jcr->plugin->file, io.status);
	    ret = bRC_Error;
         }
         if (plug_func(jcr->plugin)->endVerifyFile) {
            if (plug_func(jcr->plugin)->endVerifyFile(jcr->plugin_ctx) != bRC_OK) {
	        Dmsg2(50, "Plugin %s in endVerifyFile() status=%d\n", jcr->plugin->file, io.status);
		ret = bRC_Error;
            }
         }
      }
   }
   jcr->plugin_ctx = NULL;
   jcr->plugin = NULL;
   return ret;
}


/**
 * @brief Verify Data Plugin function to update a stream. For each block of the file, we call this function
 *
 * @param JCR *jcr
 * @param char *data   Backup data for the given file
 * @param int size     Size of the data block
 * @return bRC bRC_Error when a plugin has returned an error during the pluginIO(write)
 *             bRC_OK when all plugins are OK
 */
bRC plugin_verify_data_update(JCR *jcr, char *data, int size)
{
   bpContext *ctx;
   bacula_ctx *bctx;
   struct io_pkt io;
   bRC ret = bRC_OK;
   
   if (!jcr->plugin_verify) {
      return bRC_Skip;              // Nothing registered
   }
   foreach_alist(ctx, jcr->plugin_verify) {
      if (!is_ctx_good(ctx, jcr, bctx)) {
         return bRC_Error;
      }
      jcr->plugin_ctx = ctx;
      jcr->plugin = bctx->plugin;
      if (is_plugin_disabled(jcr)) {
         continue;
      }
      if (bctx->verifyFileCalled) {
         io.pkt_size = sizeof(io);
         io.pkt_end = sizeof(io);
         io.func = IO_WRITE;
         io.count = size;
         io.buf = data;
         io.fname = NULL;
         io.flags = 0;
         io.mode = O_WRONLY;
         io.win32 = false;
         io.lerror = 0;
         io.status = -1;
         ret = plug_func(jcr->plugin)->pluginIO(jcr->plugin_ctx, &io);
         if (io.win32) {
            errno = b_errno_win32;
         } else {
            errno = io.io_errno;
         }
         if (io.status < 0) {
            ret = bRC_Error;
         }
      }
   }
   jcr->plugin_ctx = NULL;
   jcr->plugin = NULL;
   return ret;
}

/*
 * @brief Bacula function called for each new file in verify data job
 * @param JCR *jcr
 * @param ATTR *attr  Attribute of the current file
 * @return bRC  bRC_Skip  if the list of plugins is empty
 *              bRC_Error if one of the plugin is incorrect
 *              bRC_OK    if OK
 */
bRC plugin_verify_data_open(JCR *jcr,  ATTR *attr)
{
   bpContext *ctx;
   restore_pkt rp;
   bacula_ctx *bctx;
   struct io_pkt io;
   if (!jcr->plugin_verify) {
      return bRC_Skip;              // Nothing registered
   }
   foreach_alist(ctx, jcr->plugin_verify) {
      if (!is_ctx_good(ctx, jcr, bctx)) {
         return bRC_Error;
      }
      jcr->plugin_ctx = ctx;
      jcr->plugin = bctx->plugin;
      bctx->verifyFileCalled = false;
      if (is_plugin_disabled(jcr)) {
         continue;
      }
      fill_restore_pkt(jcr, attr, &rp);
      if (plug_func(jcr->plugin)->startVerifyFile) {
         if (plug_func(jcr->plugin)->startVerifyFile(jcr->plugin_ctx, &rp) == bRC_OK) {
            io.pkt_size = sizeof(io);
            io.pkt_end = sizeof(io);
            io.func = IO_OPEN;
            io.count = 0;
            io.buf = NULL;
            io.fname = attr->fname;
            io.flags = 0;
            io.mode = O_WRONLY;
            io.win32 = false;
            io.lerror = 0;
            io.status = -1;
            bRC ret = plug_func(jcr->plugin)->pluginIO(jcr->plugin_ctx, &io);
            if (io.win32) {
               errno = b_errno_win32;
            } else {
               errno = io.io_errno;
            }
            if (ret == bRC_OK && io.status >= 0) { // -1 in case of error
               bctx->verifyFileCalled = true;
            }
         }
      }
   }
   jcr->plugin_ctx = NULL;
   jcr->plugin = NULL;
   return bRC_OK;
}

/*
 * @brief Bacula function to append a plugin registered to get a copy of the data
 *        during a Verify Data job
 * @param bpContext *ctx
 * @return void
 */
static void plugin_register_verify_data(bpContext *ctx)
{
   JCR *jcr;
   bacula_ctx *bctx;
   if (!is_ctx_good(ctx, jcr, bctx)) {
      return;
   }
   if (!jcr->is_JobLevel(L_VERIFY_DATA)) {
      return;              /* Cannot register for non DATA job */
   }
   if (!jcr->plugin_verify) {
      jcr->plugin_verify = New(alist(5, not_owned_by_alist));
   }
   jcr->plugin_verify->append(ctx);
   /* TODO: check in me->plugins if we have options for this plugin */
}


#ifdef TEST_PROGRAM

int     (*plugin_bopen)(JCR *jcr, const char *fname, uint64_t flags, mode_t mode) = NULL;
int     (*plugin_bclose)(JCR *jcr) = NULL;
ssize_t (*plugin_bread)(JCR *jcr, void *buf, size_t count) = NULL;
ssize_t (*plugin_bwrite)(JCR *jcr, void *buf, size_t count) = NULL;
boffset_t (*plugin_blseek)(JCR *jcr, boffset_t offset, int whence) = NULL;

int save_file(JCR *jcr, FF_PKT *ff_pkt, bool top_level)
{
   return 0;
}

bool set_cmd_plugin(BFILE *bfd, JCR *jcr)
{
   return true;
}

int main(int argc, char *argv[])
{
   char plugin_dir[1000];
   JCR mjcr1, mjcr2;
   JCR *jcr1 = &mjcr1;
   JCR *jcr2 = &mjcr2;

   strcpy(my_name, "test-fd");

   getcwd(plugin_dir, sizeof(plugin_dir)-1);
   load_fd_plugins(plugin_dir);

   jcr1->JobId = 111;
   new_plugins(jcr1);

   jcr2->JobId = 222;
   new_plugins(jcr2);

   generate_plugin_event(jcr1, bEventJobStart, (void *)"Start Job 1");
   generate_plugin_event(jcr1, bEventJobEnd);
   generate_plugin_event(jcr2, bEventJobStart, (void *)"Start Job 2");
   free_plugins(jcr1);
   generate_plugin_event(jcr2, bEventJobEnd);
   free_plugins(jcr2);

   unload_plugins();

   Dmsg0(dbglvl, "bacula: OK ...\n");
   close_memory_pool();
   sm_dump(false);     /* unit test */
   return 0;
}

#endif /* TEST_PROGRAM */
